/**
 * Background Service Worker
 * 负责处理API调用和数据持久化
 */

// 配置
const API_CONFIG = {
  // 旧的上传端点（兼容）
  UPLOAD_ENDPOINT: 'https://njdk.xyz:18012/api/upload-asins',
  // UPLOAD_ENDPOINT: 'http://localhost:4500/api/upload-asins',
  
  // 新的采集记录端点（推荐使用）
  COLLECTION_ENDPOINT: 'https://njdk.xyz:18012/api/collection/upload',
  // COLLECTION_ENDPOINT: 'http://localhost:4500/api/collection/upload',
  
  // 是否使用新的采集记录API
  USE_COLLECTION_API: true,

  TIMEOUT: 30000,

  // 更新检查配置
  UPDATE_CHECK_URL: 'https://zxhgs.com/extension-latest/updates.json',
  UPDATE_CHECK_INTERVAL: 24 * 60 * 60 * 1000 // 24小时检查一次
}

/**
 * 获取当前扩展版本
 * @returns {string} 版本号
 */
function getCurrentVersion() {
  try {
    return chrome.runtime.getManifest().version
  } catch (error) {
    console.error('[Background] 获取版本号失败:', error)
    return '0.0.0'
  }
}

// 存储当前状态
let currentState = {
  status: 'idle', // idle, scraping, uploading, completed, error
  currentPage: 0,
  totalPages: 0,
  asinCount: 0,
  error: null,
  lastUpdate: null
}

/**
 * 获取商家详情页面并解析
 * @param {string} url - 商家详情页面URL
 * @param {string} site - 站点代码
 * @param {string} sellerId - 商家ID
 * @returns {Promise<Object>} 解析后的商家信息
 */
async function fetchSellerDetail(url, site, sellerId) {
  console.log(`[Background] 获取商家详情: ${url}`)

  try {
    // 方法1: 尝试直接fetch（不设置User-Agent，避免CORS问题）
    try {
      const controller = new AbortController()
      const timeoutId = setTimeout(() => controller.abort(), 30000) // 30秒超时

      const response = await fetch(url, {
        method: 'GET',
        // 移除User-Agent header，在Service Worker中可能不被允许
        signal: controller.signal
      })

      clearTimeout(timeoutId)

      if (!response.ok) {
        throw new Error(`HTTP错误 ${response.status}: ${response.statusText}`)
      }

      const html = await response.text()
      console.log(`[Background] 页面获取成功（fetch方法），HTML长度: ${html.length}`)

      // 解析HTML提取商家信息
      const sellerData = parseSellerDetailHtml(html)

      return sellerData
    } catch (fetchError) {
      // 如果fetch失败，尝试使用chrome.tabs API方法
      console.log(`[Background] Fetch方法失败，尝试使用标签页方法: ${fetchError.message}`)

      // 方法2: 使用chrome.tabs创建临时标签页获取内容
      return await fetchSellerDetailViaTab(url, site, sellerId)
    }
  } catch (error) {
    console.error('[Background] 获取商家详情失败:', error)
    if (error.name === 'AbortError') {
      throw new Error('请求超时（30秒）')
    }
    throw error
  }
}

/**
 * 通过创建临时标签页获取商家详情（备用方法）
 * @param {string} url - 商家详情页面URL
 * @param {string} site - 站点代码
 * @param {string} sellerId - 商家ID
 * @returns {Promise<Object>} 解析后的商家信息
 */
async function fetchSellerDetailViaTab(url, site, sellerId) {
  return new Promise((resolve, reject) => {
    // 创建临时标签页（后台加载，不显示）
    chrome.tabs.create({
      url: url,
      active: false
    }, (tab) => {
      if (chrome.runtime.lastError) {
        reject(new Error(`创建标签页失败: ${chrome.runtime.lastError.message}`))
        return
      }

      const tabId = tab.id
      let timeoutId = null

      // 监听标签页更新
      const onUpdated = (updatedTabId, changeInfo) => {
        if (updatedTabId !== tabId) return

        // 检查是否有错误（比如404）
        if (changeInfo.status === 'complete') {
          // 检查URL是否改变（可能被重定向到错误页面）
          chrome.tabs.get(tabId, (tab) => {
            if (chrome.runtime.lastError) {
              chrome.tabs.onUpdated.removeListener(onUpdated)
              if (timeoutId) clearTimeout(timeoutId)
              reject(new Error(`获取标签页信息失败: ${chrome.runtime.lastError.message}`))
              return
            }

            // 检查是否是错误页面
            if (tab.url && (tab.url.includes('error') || tab.url.includes('notfound'))) {
              chrome.tabs.onUpdated.removeListener(onUpdated)
              if (timeoutId) clearTimeout(timeoutId)
              chrome.tabs.remove(tabId, () => {})
              reject(new Error('页面不存在或无法访问'))
              return
            }

            // 清除超时
            if (timeoutId) {
              clearTimeout(timeoutId)
            }

            // 移除监听器
            chrome.tabs.onUpdated.removeListener(onUpdated)

            // 等待一小段时间确保页面完全渲染
            setTimeout(() => {
              // 注入脚本获取页面HTML
              chrome.scripting.executeScript({
                target: { tabId: tabId },
                func: () => {
                  return document.documentElement.outerHTML
                }
              }, (results) => {
                // 关闭临时标签页
                chrome.tabs.remove(tabId, () => {
                  if (chrome.runtime.lastError) {
                    console.warn('[Background] 关闭标签页失败:', chrome.runtime.lastError)
                  }
                })

                if (chrome.runtime.lastError) {
                  reject(new Error(`执行脚本失败: ${chrome.runtime.lastError.message}`))
                  return
                }

                if (!results || !results[0] || !results[0].result) {
                  reject(new Error('无法获取页面内容'))
                  return
                }

                const html = results[0].result
                console.log(`[Background] 页面获取成功（标签页方法），HTML长度: ${html.length}`)

                try {
                  // 解析HTML提取商家信息
                  const sellerData = parseSellerDetailHtml(html)
                  resolve(sellerData)
                } catch (parseError) {
                  reject(new Error(`解析HTML失败: ${parseError.message}`))
                }
              })
            }, 2000) // 等待2秒确保页面加载完成
          })
        }
      }

      chrome.tabs.onUpdated.addListener(onUpdated)

      // 设置超时（30秒）
      timeoutId = setTimeout(() => {
        chrome.tabs.onUpdated.removeListener(onUpdated)
        chrome.tabs.remove(tabId, () => {
          if (chrome.runtime.lastError) {
            console.warn('[Background] 关闭标签页失败:', chrome.runtime.lastError)
          }
        })
        reject(new Error('请求超时（30秒）'))
      }, 30000)
    })
  })
}

/**
 * 解析商家详情HTML
 * @param {string} html - HTML内容
 * @returns {Object} 解析后的商家信息
 */
function parseSellerDetailHtml(html) {
  const data = {
    businessName: null,
    businessType: null,
    tradeRegisterNumber: null,
    phone: null,
    email: null,
    address: []
  }

  try {
    // 提取商家详情区域（根据提供的HTML结构）
    const sellerInfoMatch = html.match(/<div[^>]*id="page-section-detail-seller-info"[^>]*>([\s\S]*?)<\/div>\s*<\/div>\s*<\/div>\s*<\/div>/i)
    const content = sellerInfoMatch ? sellerInfoMatch[1] : html

    // 提取Business Name - 格式: <span class="a-text-bold">Business Name:</span><span>dongguanshichurongshangmaoyouxiangongsi</span>
    const businessNamePatterns = [
      /Business Name:\s*<\/span>\s*<span[^>]*>([^<]+)<\/span>/i,
      /<span[^>]*class="[^"]*a-text-bold[^"]*"[^>]*>Business Name:\s*<\/span>\s*<span[^>]*>([^<]+)<\/span>/i
    ]
    for (const pattern of businessNamePatterns) {
      const match = content.match(pattern)
      if (match && match[1]) {
        data.businessName = match[1].trim()
        break
      }
    }

    // 提取Business Type
    const businessTypePatterns = [
      /Business Type:\s*<\/span>\s*<span[^>]*>([^<]+)<\/span>/i,
      /<span[^>]*class="[^"]*a-text-bold[^"]*"[^>]*>Business Type:\s*<\/span>\s*<span[^>]*>([^<]+)<\/span>/i
    ]
    for (const pattern of businessTypePatterns) {
      const match = content.match(pattern)
      if (match && match[1]) {
        data.businessType = match[1].trim()
        break
      }
    }

    // 提取Trade Register Number
    const tradeRegisterPatterns = [
      /Trade Register Number:\s*<\/span>\s*<span[^>]*>([^<]+)<\/span>/i,
      /<span[^>]*class="[^"]*a-text-bold[^"]*"[^>]*>Trade Register Number:\s*<\/span>\s*<span[^>]*>([^<]+)<\/span>/i
    ]
    for (const pattern of tradeRegisterPatterns) {
      const match = content.match(pattern)
      if (match && match[1]) {
        data.tradeRegisterNumber = match[1].trim()
        break
      }
    }

    // 提取Phone number
    const phonePatterns = [
      /Phone number:\s*<\/span>\s*<span[^>]*>([^<]+)<\/span>/i,
      /<span[^>]*class="[^"]*a-text-bold[^"]*"[^>]*>Phone number:\s*<\/span>\s*<span[^>]*>([^<]+)<\/span>/i
    ]
    for (const pattern of phonePatterns) {
      const match = content.match(pattern)
      if (match && match[1]) {
        data.phone = match[1].trim()
        break
      }
    }

    // 提取Email
    const emailPatterns = [
      /Email:\s*<\/span>\s*<span[^>]*>([^<]+)<\/span>/i,
      /<span[^>]*class="[^"]*a-text-bold[^"]*"[^>]*>Email:\s*<\/span>\s*<span[^>]*>([^<]+)<\/span>/i
    ]
    for (const pattern of emailPatterns) {
      const match = content.match(pattern)
      if (match && match[1]) {
        data.email = match[1].trim()
        break
      }
    }

    // 提取Business Address（多行，使用indent-left类）
    // 格式: <div class="a-row a-spacing-none indent-left"><span>长安镇</span></div>
    const addressPatterns = [
      // 提取Business Address之后的所有indent-left div
      /Business Address:\s*<\/span>\s*<\/div>([\s\S]*?)(?:This seller|<\/div>\s*<\/div>\s*<\/div>)/i,
      /Business Address:[\s\S]*?<div[^>]*class="[^"]*indent-left[^"]*"[^>]*>([\s\S]*?)(?:This seller|<\/div>\s*<\/div>\s*<\/div>)/i
    ]

    for (const pattern of addressPatterns) {
      const addressSectionMatch = content.match(pattern)
      if (addressSectionMatch) {
        const addressSection = addressSectionMatch[1] || addressSectionMatch[0]
        // 提取所有indent-left div中的span内容
        const addressDivMatches = addressSection.matchAll(/<div[^>]*class="[^"]*indent-left[^"]*"[^>]*>\s*<span[^>]*>([^<]+)<\/span>\s*<\/div>/gi)
        for (const match of addressDivMatches) {
          const addressLine = match[1].trim()
          if (addressLine && !addressLine.match(/^This seller/i)) {
            data.address.push(addressLine)
          }
        }
        if (data.address.length > 0) break
      }
    }

    // 如果还是没找到，尝试更宽松的匹配
    if (data.address.length === 0) {
      const allIndentLeftMatches = content.matchAll(/<div[^>]*class="[^"]*indent-left[^"]*"[^>]*>\s*<span[^>]*>([^<]+)<\/span>/gi)
      for (const match of allIndentLeftMatches) {
        const addressLine = match[1].trim()
        if (addressLine && !addressLine.match(/^This seller/i)) {
          data.address.push(addressLine)
        }
      }
    }

    console.log('[Background] 解析商家信息完成:', data)
    return data
  } catch (error) {
    console.error('[Background] 解析HTML失败:', error)
    throw new Error(`解析商家信息失败: ${error.message}`)
  }
}

/**
 * 查询商家统计信息
 * @param {string} apiEndpoint - API基础地址
 * @param {number} minAsinCount - 最小ASIN数量
 * @param {number} minOfferCount - 最小报价数量
 * @returns {Promise<Object>} API响应
 */
async function querySellerStatsFromApi(apiEndpoint, minAsinCount, minOfferCount) {
  console.log(`[Background] 查询商家统计: ASIN数量>${minAsinCount}, 报价数量>${minOfferCount}`)

  try {
    // 构建查询API地址（从上传地址推导）
    const baseUrl = apiEndpoint.replace('/api/upload-asins', '')
    const queryUrl = `${baseUrl}/api/seller-stats`

    console.log(`[Background] 查询URL: ${queryUrl}`)

    const controller = new AbortController()
    const timeoutId = setTimeout(() => controller.abort(), API_CONFIG.TIMEOUT)

    // 构建查询参数
    const queryParams = new URLSearchParams({
      minAsinCount: minAsinCount.toString(),
      minOfferCount: minOfferCount.toString()
    })

    const fullUrl = `${queryUrl}?${queryParams.toString()}`
    console.log(`[Background] 完整查询URL: ${fullUrl}`)

    let response
    try {
      response = await fetch(fullUrl, {
        method: 'GET',
        headers: {
          'Content-Type': 'application/json'
        },
        signal: controller.signal
      })
    } catch (fetchError) {
      clearTimeout(timeoutId)
      console.error('[Background] 网络请求失败:', fetchError)

      if (fetchError.name === 'AbortError') {
        throw new Error(`请求超时（${API_CONFIG.TIMEOUT / 1000}秒）`)
      } else if (fetchError.message.includes('Failed to fetch')) {
        throw new Error(
          `无法连接到服务器 ${queryUrl}。请检查：\n1. 服务器是否运行\n2. 网络连接\n3. API地址是否正确`
        )
      } else {
        throw new Error(`网络错误: ${fetchError.message}`)
      }
    }

    clearTimeout(timeoutId)

    console.log('[Background] API响应状态:', response.status, response.statusText)

    if (!response.ok) {
      let errorText
      try {
        errorText = await response.text()
        console.error('[Background] API错误响应:', errorText)
      } catch (e) {
        errorText = '无法读取错误信息'
      }
      throw new Error(`API返回错误 ${response.status}: ${response.statusText}\n${errorText}`)
    }

    let result
    try {
      result = await response.json()
      console.log('[Background] 商家统计查询成功:', result)
    } catch (jsonError) {
      console.error('[Background] 解析响应JSON失败:', jsonError)
      throw new Error('服务器响应格式错误')
    }

    return result
  } catch (error) {
    console.error('[Background] 商家统计查询失败:', error)
    throw error
  }
}

/**
 * 后台智能采集所有页面（并发优化版）
 * @param {string} baseUrl - 基础URL
 * @param {number} totalPages - 总页数
 * @param {Object} pageInfo - 页面信息
 */
async function startBackgroundScraping(baseUrl, totalPages, pageInfo) {
  console.log(`[Background] 后台智能采集开始: ${totalPages} 页`)
  
  // 获取并发数配置
  const { concurrency } = await chrome.storage.local.get('concurrency')
  const maxConcurrency = concurrency || 5
  console.log(`[Background] 并发数: ${maxConcurrency}`)
  
  const allAsins = new Set()
  let successCount = 0
  let failCount = 0
  let completedCount = 0
  
  // 更新状态
  updateState({
    status: 'scraping',
    currentPage: 0,
    totalPages: totalPages,
    asinCount: 0
  })
  
  // 构建页面URL列表
  const pageUrls = []
  const urlObj = new URL(baseUrl)
  
  for (let page = 1; page <= Math.min(totalPages, 50); page++) { // 限制最多50页
    const pageUrl = new URL(baseUrl)
    pageUrl.searchParams.set('page', page.toString())
    pageUrls.push({
      page: page,
      url: pageUrl.href
    })
  }
  
  console.log(`[Background] 准备采集 ${pageUrls.length} 个页面（${maxConcurrency}个并发）`)
  
  const startTime = Date.now()
  
  // 并发采集函数
  const scrapePage = async (pageData) => {
    try {
      console.log(`[Background] [并发] 采集第 ${pageData.page} 页...`)
      
      // 在后台标签页中采集
      const asins = await scrapePageInBackground(pageData.url)
      
      // 添加到总集合（注意并发安全）
      asins.forEach(asin => allAsins.add(asin))
      
      successCount++
      completedCount++
      
      console.log(`[Background] [并发] 第 ${pageData.page} 页完成，获取 ${asins.length} 个ASIN，累计 ${allAsins.size} 个 (${completedCount}/${pageUrls.length})`)
      
      // 更新进度
      updateState({
        status: 'scraping',
        currentPage: completedCount,
        totalPages: totalPages,
        asinCount: allAsins.size
      })
      
      return { success: true, page: pageData.page, count: asins.length }
      
    } catch (error) {
      console.error(`[Background] [并发] 第 ${pageData.page} 页采集失败:`, error)
      failCount++
      completedCount++
      
      // 更新进度（即使失败也要更新）
      updateState({
        status: 'scraping',
        currentPage: completedCount,
        totalPages: totalPages,
        asinCount: allAsins.size
      })
      
      return { success: false, page: pageData.page, error: error.message }
    }
  }
  
  // 并发控制：分批处理
  const results = []
  for (let i = 0; i < pageUrls.length; i += maxConcurrency) {
    const batch = pageUrls.slice(i, i + maxConcurrency)
    console.log(`[Background] 处理批次 ${Math.floor(i / maxConcurrency) + 1}/${Math.ceil(pageUrls.length / maxConcurrency)}: 第${batch[0].page}-${batch[batch.length - 1].page}页`)
    
    // 并发执行当前批次
    const batchResults = await Promise.all(batch.map(scrapePage))
    results.push(...batchResults)
    
    // 批次间短暂延迟（避免请求过快）
    if (i + maxConcurrency < pageUrls.length) {
      await new Promise(resolve => setTimeout(resolve, 500))
    }
  }
  
  const endTime = Date.now()
  const duration = ((endTime - startTime) / 1000).toFixed(1)
  const speed = (allAsins.size / (duration / 60)).toFixed(0)
  
  console.log(`[Background] ========== 后台采集完成 ==========`)
  console.log(`[Background] 成功: ${successCount}, 失败: ${failCount}, 总ASIN: ${allAsins.size}`)
  console.log(`[Background] 耗时: ${duration}秒, 速度: ${speed} ASIN/分钟`)
  console.log(`[Background] 并发数: ${maxConcurrency}`)
  
  // 上传到API
  updateState({
    status: 'uploading',
    currentPage: pageUrls.length,
    totalPages: totalPages,
    asinCount: allAsins.size
  })
  
  try {
    await uploadAsinsToApi(Array.from(allAsins), pageInfo)
    
    updateState({
      status: 'completed',
      currentPage: pageUrls.length,
      totalPages: totalPages,
      asinCount: allAsins.size,
      error: null
    })
    
    // 通知完成
    chrome.runtime.sendMessage({
      type: 'UPLOAD_COMPLETE',
      data: {
        success: true,
        asinCount: allAsins.size,
        pageInfo: pageInfo
      }
    }, () => {
      if (chrome.runtime.lastError) {
        // 忽略
      }
    })
    
  } catch (error) {
    console.error('[Background] 上传失败:', error)
    updateState({
      status: 'error',
      error: `上传失败: ${error.message}`
    })
    
    chrome.runtime.sendMessage({
      type: 'UPLOAD_ERROR',
      data: { error: error.message }
    }, () => {
      if (chrome.runtime.lastError) {
        // 忽略
      }
    })
  }
}

/**
 * 在后台标签页中采集单个页面
 * @param {string} url - 页面URL
 * @returns {Promise<string[]>} ASIN数组
 */
async function scrapePageInBackground(url) {
  return new Promise((resolve, reject) => {
    // 创建后台标签页（不激活）
    chrome.tabs.create({
      url: url,
      active: false // 不激活，在后台打开
    }, (tab) => {
      if (chrome.runtime.lastError) {
        reject(new Error(`创建标签页失败: ${chrome.runtime.lastError.message}`))
        return
      }
      
      const tabId = tab.id
      let timeoutId = null
      
      // 监听标签页加载完成
      const onUpdated = (updatedTabId, changeInfo) => {
        if (updatedTabId !== tabId) return
        
        if (changeInfo.status === 'complete') {
          // 清除超时
          if (timeoutId) {
            clearTimeout(timeoutId)
          }
          
          // 移除监听
          chrome.tabs.onUpdated.removeListener(onUpdated)
          
          // 等待页面渲染
          setTimeout(() => {
            // 执行采集脚本
            chrome.scripting.executeScript({
              target: { tabId: tabId },
              func: () => {
                // 提取ASIN
                const asins = new Set()
                const elements = document.querySelectorAll('[data-asin]')
                
                elements.forEach(el => {
                  const asin = el.getAttribute('data-asin')
                  // 验证ASIN格式
                  if (asin && /^[A-Z0-9]{10}$/.test(asin) && !/^\d{10}$/.test(asin)) {
                    asins.add(asin)
                  }
                })
                
                return Array.from(asins)
              }
            }, (results) => {
              // 关闭后台标签页
              chrome.tabs.remove(tabId, () => {
                if (chrome.runtime.lastError) {
                  console.warn('[Background] 关闭标签页失败:', chrome.runtime.lastError)
                }
              })
              
              if (chrome.runtime.lastError) {
                reject(new Error(`执行脚本失败: ${chrome.runtime.lastError.message}`))
                return
              }
              
              if (!results || !results[0] || !results[0].result) {
                reject(new Error('无法获取页面内容'))
                return
              }
              
              const asins = results[0].result
              resolve(asins)
            })
          }, 2000) // 等待2秒确保页面加载完成
        }
      }
      
      chrome.tabs.onUpdated.addListener(onUpdated)
      
      // 设置超时（30秒）
      timeoutId = setTimeout(() => {
        chrome.tabs.onUpdated.removeListener(onUpdated)
        chrome.tabs.remove(tabId, () => {
          if (chrome.runtime.lastError) {
            console.warn('[Background] 关闭标签页失败:', chrome.runtime.lastError)
          }
        })
        reject(new Error('页面加载超时（30秒）'))
      }, 30000)
    })
  })
}

/**
 * 上传ASIN到后端API（包含完整页面信息）
 * 支持新旧两种API格式
 * @param {string[]} asins - ASIN数组
 * @param {Object} pageInfo - 页面信息对象
 * @returns {Promise<Object>} API响应
 */
async function uploadAsinsToApi(asins, pageInfo = {}) {
  console.log(`[Background] 准备上传 ${asins.length} 个ASIN到API`)
  console.log(`[Background] 页面信息:`, pageInfo)
  console.log(`[Background] 使用${API_CONFIG.USE_COLLECTION_API ? '新' : '旧'}API端点`)

  try {
    // 获取当前登录用户信息
    let collectedBy = 'browser-extension'
    try {
      const { currentUser } = await chrome.storage.local.get('currentUser')
      if (currentUser && currentUser.username) {
        collectedBy = currentUser.username
        console.log(`[Background] 采集人: ${collectedBy}`)
      }
    } catch (error) {
      console.warn('[Background] 无法获取当前用户信息:', error)
    }

    const controller = new AbortController()
    const timeoutId = setTimeout(() => controller.abort(), API_CONFIG.TIMEOUT)

    let uploadData, endpoint

    if (API_CONFIG.USE_COLLECTION_API) {
      // 新的采集记录API格式
      endpoint = API_CONFIG.COLLECTION_ENDPOINT
      uploadData = {
        // ASIN列表（必需）
        asins: asins,

        // 站点信息（必需）
        site: pageInfo.site || 'US',
        
        // 采集人信息（使用登录用户名）
        collectedBy: collectedBy,
        
        // 批次信息（可选，如果不提供会自动生成）
        batchName: `Browser Collection ${new Date().toISOString().split('T')[0]}`,
        
        // 页面信息对象（完整传递，controller会处理）
        pageInfo: {
          ...pageInfo,
          // 确保有完整的分类信息
          categoryPath: pageInfo.categoryPath || pageInfo.category,
          categoryBreadcrumb: pageInfo.categoryBreadcrumb || [],
          categoryId: pageInfo.categoryId || null,
          searchKeyword: pageInfo.searchKeyword || null,
          url: pageInfo.url || null
        },

        // 额外数据
        extraData: {
          timestamp: new Date().toISOString(),
          source: 'amazon-asin-scraper-v2',
          version: getCurrentVersion(),
          marketplace: pageInfo.marketplace || null,
          totalResults: pageInfo.totalResults || null,
          sellerId: pageInfo.sellerId || null,
          sellerName: pageInfo.sellerName || null
        }
      }

      console.log('[Background] 使用新API格式（采集记录）')
    } else {
      // 旧的API格式（兼容）
      endpoint = API_CONFIG.UPLOAD_ENDPOINT
      uploadData = {
        asinList: asins,
        timestamp: new Date().toISOString(),
        source: 'amazon-asin-scraper',
        version: '2.0.1',
        site: pageInfo.site || 'UNKNOWN',
        marketplace: pageInfo.marketplace || null,
        url: pageInfo.url || null,
        category: pageInfo.category || null,
        categoryId: pageInfo.categoryId || null,
        searchKeyword: pageInfo.searchKeyword || null,
        totalResults: pageInfo.totalResults || null,
        sellerId: pageInfo.sellerId || null,
        sellerName: pageInfo.sellerName || null,
        collectionStats: {
          totalAsins: asins.length,
          collectedAt: pageInfo.timestamp || new Date().toISOString()
        }
      }

      console.log('[Background] 使用旧API格式（兼容模式）')
    }

    console.log('[Background] 上传数据结构:', JSON.stringify(uploadData, null, 2))
    console.log('[Background] 发送请求到:', endpoint)
    console.log('[Background] 请求方法: POST')
    console.log('[Background] 数据大小:', JSON.stringify(uploadData).length, 'bytes')

    let response
    try {
      response = await fetch(endpoint, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json'
        },
        body: JSON.stringify(uploadData),
        signal: controller.signal
      })
    } catch (fetchError) {
      clearTimeout(timeoutId)
      console.error('[Background] 网络请求失败:', fetchError)

      // 提供更详细的错误信息
      if (fetchError.name === 'AbortError') {
        throw new Error(`请求超时（${API_CONFIG.TIMEOUT / 1000}秒）`)
      } else if (fetchError.message.includes('Failed to fetch')) {
        throw new Error(`无法连接到服务器 ${endpoint}。请检查：\n1. 服务器是否运行\n2. 网络连接\n3. HTTPS证书是否有效`)
      } else {
        throw new Error(`网络错误: ${fetchError.message}`)
      }
    }

    clearTimeout(timeoutId)

    console.log('[Background] API响应状态:', response.status, response.statusText)

    if (!response.ok) {
      let errorText
      try {
        errorText = await response.text()
        console.error('[Background] API错误响应:', errorText)
      } catch (e) {
        errorText = '无法读取错误信息'
      }
      throw new Error(`API返回错误 ${response.status}: ${response.statusText}\n${errorText}`)
    }

    let result
    try {
      result = await response.json()
      console.log('[Background] ASIN上传成功:', result)
    } catch (jsonError) {
      console.error('[Background] 解析响应JSON失败:', jsonError)
      throw new Error('服务器响应格式错误')
    }

    return result
  } catch (error) {
    console.error('[Background] ASIN上传失败:', error)
    throw error
  }
}

/**
 * 比较版本号
 * @param {string} v1 - 版本号1
 * @param {string} v2 - 版本号2
 * @returns {number} - 1: v1 > v2, 0: v1 = v2, -1: v1 < v2
 */
function compareVersions(v1, v2) {
  const parts1 = v1.split('.').map(Number)
  const parts2 = v2.split('.').map(Number)

  for (let i = 0; i < Math.max(parts1.length, parts2.length); i++) {
    const part1 = parts1[i] || 0
    const part2 = parts2[i] || 0

    if (part1 > part2) return 1
    if (part1 < part2) return -1
  }

  return 0
}

/**
 * 检查更新
 * @returns {Promise<Object|null>} 更新信息，如果没有更新则返回null
 */
async function checkForUpdates() {
  console.log('[Background] 检查扩展更新...')
  const currentVersion = getCurrentVersion()
  console.log(`[Background] 当前版本: ${currentVersion}`)

  try {
    const controller = new AbortController()
    const timeoutId = setTimeout(() => controller.abort(), 10000) // 10秒超时

    const response = await fetch(API_CONFIG.UPDATE_CHECK_URL, {
      method: 'GET',
      cache: 'no-cache',
      signal: controller.signal
    })

    clearTimeout(timeoutId)

    if (!response.ok) {
      throw new Error(`HTTP ${response.status}: ${response.statusText}`)
    }

    const updateInfo = await response.json()
    console.log('[Background] 服务器最新版本:', updateInfo.latest_version)

    // 比较版本号
    const comparison = compareVersions(updateInfo.latest_version, currentVersion)

    if (comparison > 0) {
      // 有新版本
      console.log('[Background] 🎉 发现新版本:', updateInfo.latest_version)

      // 保存更新信息
      await chrome.storage.local.set({
        updateAvailable: true,
        updateInfo: updateInfo,
        lastUpdateCheck: Date.now()
      })

      // 通知popup（如果打开）
      chrome.runtime.sendMessage({
        type: 'UPDATE_AVAILABLE',
        data: updateInfo
      }, () => {
        if (chrome.runtime.lastError) {
          // 忽略错误（popup可能未打开）
        }
      })

      return updateInfo
    } else {
      console.log('[Background] ✅ 当前已是最新版本')

      // 更新检查时间
      await chrome.storage.local.set({
        updateAvailable: false,
        lastUpdateCheck: Date.now()
      })

      return null
    }
  } catch (error) {
    console.error('[Background] 检查更新失败:', error)
    return null
  }
}

/**
 * 定期检查更新
 */
function scheduleUpdateCheck() {
  // 立即检查一次
  checkForUpdates()

  // 设置定时检查
  setInterval(() => {
    checkForUpdates()
  }, API_CONFIG.UPDATE_CHECK_INTERVAL)
}

/**
 * 更新状态
 * @param {Object} newState - 新状态
 */
function updateState(newState) {
  currentState = {
    ...currentState,
    ...newState,
    lastUpdate: Date.now()
  }

  // 保存到storage
  chrome.storage.local.set({ state: currentState })

  // 通知所有popup实例
  chrome.runtime.sendMessage(
    {
      type: 'STATE_UPDATE',
      data: currentState
    },
    () => {
      // 检查并忽略错误（Popup可能未打开）
      if (chrome.runtime.lastError) {
        // 忽略：Popup未打开时的错误
      }
    }
  )
}

/**
 * 处理来自content script的消息
 */
chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
  console.log('[Background] 收到消息:', message.type, message)

  switch (message.type) {
    case 'SCRAPING_PROGRESS':
      updateState({
        status: 'scraping',
        currentPage: message.data.currentPage,
        totalPages: message.data.totalPages,
        asinCount: message.data.asinCount
      })
      sendResponse({ success: true })
      break

    case 'SCRAPING_COMPLETE':
      console.log(`[Background] 采集完成，获得 ${message.data.totalCount} 个ASIN`)

      // 更新状态为上传中
      updateState({
        status: 'uploading',
        asinCount: message.data.totalCount,
        totalPages: message.data.totalPages
      })

      // 上传到API（包含页面信息）
      uploadAsinsToApi(message.data.asins, message.data.pageInfo)
        .then(result => {
          updateState({
            status: 'completed',
            error: null
          })

          // 发送上传完成消息
          chrome.runtime.sendMessage(
            {
              type: 'UPLOAD_COMPLETE',
              data: {
                success: true,
                asinCount: message.data.totalCount,
                pageInfo: message.data.pageInfo,
                apiResponse: result
              }
            },
            () => {
              if (chrome.runtime.lastError) {
                // 忽略错误
              }
            }
          )
        })
        .catch(error => {
          updateState({
            status: 'error',
            error: `上传失败: ${error.message}`
          })

          // 发送上传错误消息
          chrome.runtime.sendMessage(
            {
              type: 'UPLOAD_ERROR',
              data: {
                error: error.message
              }
            },
            () => {
              if (chrome.runtime.lastError) {
                // 忽略错误
              }
            }
          )
        })

      sendResponse({ success: true })
      break

    case 'SCRAPING_CANCELLED':
      console.log(`[Background] 采集已取消，获得 ${message.data.totalCount} 个ASIN`)

      // 如果有数据，仍然上传
      if (message.data.totalCount > 0) {
        updateState({
          status: 'uploading',
          asinCount: message.data.totalCount,
          totalPages: message.data.totalPages
        })

        uploadAsinsToApi(message.data.asins, message.data.pageInfo)
          .then(result => {
            updateState({
              status: 'cancelled',
              error: null
            })

            chrome.runtime.sendMessage(
              {
                type: 'UPLOAD_COMPLETE',
                data: {
                  success: true,
                  asinCount: message.data.totalCount,
                  pageInfo: message.data.pageInfo,
                  apiResponse: result
                }
              },
              () => {
                if (chrome.runtime.lastError) {
                  // 忽略错误
                }
              }
            )
          })
          .catch(error => {
            updateState({
              status: 'error',
              error: `上传失败: ${error.message}`
            })

            chrome.runtime.sendMessage(
              {
                type: 'UPLOAD_ERROR',
                data: {
                  error: error.message
                }
              },
              () => {
                if (chrome.runtime.lastError) {
                  // 忽略错误
                }
              }
            )
          })
      } else {
        updateState({
          status: 'cancelled',
          error: null
        })
      }

      sendResponse({ success: true })
      break

    case 'SCRAPING_ERROR':
      updateState({
        status: 'error',
        error: `采集失败: ${message.data.error}`
      })
      sendResponse({ success: true })
      break

    case 'GET_STATUS':
      sendResponse({ success: true, state: currentState })
      break

    case 'RESET_STATUS':
      updateState({
        status: 'idle',
        currentPage: 0,
        totalPages: 0,
        asinCount: 0,
        error: null
      })
      sendResponse({ success: true })
      break

    case 'UPDATE_API_ENDPOINT':
      API_CONFIG.UPLOAD_ENDPOINT = message.data.endpoint
      chrome.storage.local.set({ apiEndpoint: message.data.endpoint })
      sendResponse({ success: true })
      break

    case 'UPDATE_CONFIG':
      // 更新配置
      if (message.data.endpoint) {
        API_CONFIG.UPLOAD_ENDPOINT = message.data.endpoint
        API_CONFIG.COLLECTION_ENDPOINT = message.data.endpoint.replace('/upload-asins', '/collection/upload')
      }
      chrome.storage.local.set({
        apiEndpoint: message.data.endpoint,
        scrapingMode: message.data.scrapingMode,
        concurrency: message.data.concurrency || 5
      })
      console.log('[Background] 配置已更新:', message.data)
      sendResponse({ success: true })
      break

    case 'START_BACKGROUND_SCRAPING':
      console.log('[Background] 开始后台智能采集:', message.data)
      // 异步处理后台采集
      startBackgroundScraping(message.data.baseUrl, message.data.totalPages, message.data.pageInfo)
        .then(() => {
          sendResponse({ success: true })
        })
        .catch(error => {
          console.error('[Background] 后台采集失败:', error)
          sendResponse({ success: false, error: error.message })
        })
      return true // 保持消息通道开启

    case 'QUERY_SELLER_STATS':
      console.log('[Background] 处理商家统计查询请求:', message.data)
      // 异步处理查询请求
      querySellerStatsFromApi(
        message.data.apiEndpoint,
        message.data.minAsinCount,
        message.data.minOfferCount
      )
        .then(result => {
          console.log('[Background] 商家统计查询成功，返回结果:', result)
          // API返回格式: { success: true, data: { stats, sellers, pagination } }
          // 我们需要提取 data 部分
          if (result.success && result.data) {
            sendResponse({ success: true, data: result.data })
          } else {
            sendResponse({ success: false, error: 'API返回数据格式错误' })
          }
        })
        .catch(error => {
          console.error('[Background] 商家统计查询失败:', error)
          sendResponse({ success: false, error: error.message })
        })
      return true // 保持消息通道开启以支持异步响应

    case 'FETCH_SELLER_DETAIL':
      console.log('[Background] 获取商家详情:', message.data)
      // 异步处理获取商家详情
      fetchSellerDetail(message.data.url, message.data.site, message.data.sellerId)
        .then(data => {
          sendResponse({ success: true, data })
        })
        .catch(error => {
          console.error('[Background] 获取商家详情失败:', error)
          sendResponse({ success: false, error: error.message })
        })
      return true // 保持消息通道开启以支持异步响应

    case 'CHECK_UPDATE':
      console.log('[Background] 手动检查更新')
      checkForUpdates()
        .then(updateInfo => {
          if (updateInfo) {
            sendResponse({ success: true, hasUpdate: true, updateInfo })
          } else {
            sendResponse({ success: true, hasUpdate: false })
          }
        })
        .catch(error => {
          sendResponse({ success: false, error: error.message })
        })
      return true // 保持消息通道开启以支持异步响应

    default:
      console.warn('[Background] 未知消息类型:', message.type)
      sendResponse({ success: false, error: 'Unknown message type' })
      return false
  }
})

// 初始化：加载保存的配置
chrome.storage.local.get(['apiEndpoint', 'state'], result => {
  // 添加空值检查
  if (!result) {
    console.log('[Background] 无保存的配置数据')
    return
  }

  if (result.apiEndpoint) {
    API_CONFIG.UPLOAD_ENDPOINT = result.apiEndpoint
    console.log('[Background] 已加载API配置:', API_CONFIG.UPLOAD_ENDPOINT)
  }

  if (result.state) {
    currentState = result.state
    console.log('[Background] 已加载状态:', currentState)
  }

  // 启动更新检查
  scheduleUpdateCheck()
})

console.log('[Background] Service Worker已启动')
