/**
 * 分页处理工具
 */

import { SCRAPING_CONFIG } from './constants.js'

export class PaginationHandler {
  static hasNextPage() {
    const nextButton = this.getNextPageButton();
    if (!nextButton) return false;

    return !(
      nextButton.classList.contains('disabled') ||
      nextButton.classList.contains('s-pagination-disabled') ||
      nextButton.hasAttribute('disabled') ||
      nextButton.getAttribute('aria-disabled') === 'true'
    );
  }

  /**
   * 获取下一页按钮元素
   * @returns {HTMLElement|null} 下一页按钮元素
   */
  static getNextPageButton() {
    // 亚马逊底部分页选择器
    const selectors = [
      'a.s-pagination-next',
      '.s-pagination-item.s-pagination-next',
      'a[aria-label="Next"]',
      'a[aria-label="Go to next page"]',
      // 备用选择器
      '.pagination .next',
      'a.next'
    ]

    for (const selector of selectors) {
      const button = document.querySelector(selector)
      if (button) {
        console.log(`[Pagination] 找到下一页按钮: ${selector}`)
        return button
      }
    }

    console.warn('[Pagination] 未找到下一页按钮')
    return null
  }

  /**
   * 点击进入下一页
   * @returns {Promise<boolean>} 是否成功进入下一页
   */
  static async goToNextPage() {
    const nextButton = this.getNextPageButton()
    if (!nextButton) {
      console.log('[Pagination] 没有找到下一页按钮')
      return false
    }

    if (!this.hasNextPage()) {
      console.log('[Pagination] 已经是最后一页')
      return false
    }

    try {
      // 记录当前页面的特征，用于验证是否成功跳转
      const currentUrl = window.location.href
      const currentPageNumber = this.getCurrentPageNumber()

      console.log(`[Pagination] 点击下一页按钮，当前页码: ${currentPageNumber}`)

      // 点击按钮
      nextButton.click()

      // 等待页面加载
      await this.waitForPageLoad(currentUrl, currentPageNumber)

      console.log('[Pagination] 成功进入下一页')
      return true
    } catch (error) {
      console.error('[Pagination] 进入下一页失败:', error)
      return false
    }
  }

  /**
   * 获取当前页码
   * @returns {number} 当前页码
   */
  static getCurrentPageNumber() {
    // 方法1: 从亚马逊分页组件获取（当前选中的页码）
    const activePageElement = document.querySelector('.s-pagination-item.s-pagination-selected')
    if (activePageElement) {
      const pageText = activePageElement.textContent.trim()
      const pageNum = parseInt(pageText, 10)
      if (!isNaN(pageNum)) {
        console.log(`[Pagination] 当前页码: ${pageNum}`)
        return pageNum
      }
    }

    // 方法2: 从URL获取页码参数
    const urlParams = new URLSearchParams(window.location.search)
    const pageFromUrl = urlParams.get('page') || urlParams.get('p')
    if (pageFromUrl) {
      const pageNum = parseInt(pageFromUrl, 10)
      if (!isNaN(pageNum)) {
        console.log(`[Pagination] 从URL获取页码: ${pageNum}`)
        return pageNum
      }
    }

    // 方法3: 从所有页码按钮中找到当前激活的
    const allPageButtons = document.querySelectorAll('.s-pagination-item')
    for (const btn of allPageButtons) {
      if (btn.classList.contains('s-pagination-selected') ||
          btn.classList.contains('s-pagination-disabled')) {
        const pageText = btn.textContent.trim()
        const pageNum = parseInt(pageText, 10)
        if (!isNaN(pageNum)) {
          console.log(`[Pagination] 找到当前页码: ${pageNum}`)
          return pageNum
        }
      }
    }

    console.log('[Pagination] 未找到页码，默认为第1页')
    return 1
  }

  /**
   * 等待页面加载完成
   * @param {string} previousUrl - 之前的URL
   * @param {number} previousPage - 之前的页码
   * @returns {Promise<void>}
   */
  static async waitForPageLoad(previousUrl, previousPage) {
    const maxWaitTime = 10000 // 最长等待10秒
    const checkInterval = 200 // 每200ms检查一次
    let elapsedTime = 0

    return new Promise((resolve, reject) => {
      const checkPageChanged = setInterval(() => {
        elapsedTime += checkInterval

        // 检查URL是否改变或页码是否改变
        const currentUrl = window.location.href
        const currentPage = this.getCurrentPageNumber()

        if (currentUrl !== previousUrl || currentPage > previousPage) {
          clearInterval(checkPageChanged)
          // 额外等待一段时间确保内容加载
          setTimeout(() => resolve(), SCRAPING_CONFIG.PAGE_LOAD_DELAY)
          return
        }

        // 超时
        if (elapsedTime >= maxWaitTime) {
          clearInterval(checkPageChanged)
          reject(new Error('页面加载超时'))
        }
      }, checkInterval)
    })
  }

  /**
   * 获取总页数（如果可能）
   * @returns {number|null} 总页数或null
   */
  static getTotalPages() {
    // 方法1: 从亚马逊分页按钮获取（找最大的页码数字，包括禁用的）
    const pageButtons = document.querySelectorAll('.s-pagination-item')
    let maxPage = 0

    console.log(`[Pagination] 找到 ${pageButtons.length} 个分页元素`)

    pageButtons.forEach(btn => {
      // 跳过Previous、Next和省略号
      if (btn.classList.contains('s-pagination-previous') ||
          btn.classList.contains('s-pagination-next') ||
          btn.classList.contains('s-pagination-ellipsis')) {
        return
      }

      const pageText = btn.textContent.trim()
      const pageNum = parseInt(pageText, 10)

      if (!isNaN(pageNum) && pageNum > maxPage) {
        maxPage = pageNum
        console.log(`[Pagination] 发现页码: ${pageNum}`)
      }
    })

    if (maxPage > 0) {
      console.log(`[Pagination] ✓ 总页数: ${maxPage}`)
      return maxPage
    }

    // 方法2: 从页面信息文本获取
    const resultStats = document.querySelector('.s-desktop-toolbar, .s-result-count')
    if (resultStats) {
      const text = resultStats.textContent
      // 匹配 "1-16 of 498 results" 这样的文本
      const match = text.match(/of\s+([\d,]+)\s+results/i)
      if (match) {
        const totalResults = parseInt(match[1].replace(/,/g, ''), 10)
        const itemsPerPage = 16 // 亚马逊默认每页16个
        const estimatedPages = Math.ceil(totalResults / itemsPerPage)
        console.log(`[Pagination] 估算总页数: ${estimatedPages}（基于 ${totalResults} 个结果）`)
        return estimatedPages
      }
    }

    console.log('[Pagination] ⚠️ 无法确定总页数')
    return null
  }
}

