/**
 * 消息工具模块 - 统一的消息通信封装
 */

/**
 * 发送消息到 background
 */
export async function sendToBackground(type, data = {}) {
  try {
    return await chrome.runtime.sendMessage({ type, data });
  } catch (error) {
    console.error(`[Message] 发送失败 ${type}:`, error);
    throw error;
  }
}

/**
 * 发送消息到 content script（带重试）
 */
export async function sendToTab(tabId, type, data = {}, maxRetries = 3) {
  for (let i = 0; i < maxRetries; i++) {
    try {
      return await chrome.tabs.sendMessage(tabId, { type, data });
    } catch (error) {
      const isConnectionError =
        error.message.includes('Receiving end does not exist') ||
        error.message.includes('Could not establish connection');

      if (i < maxRetries - 1) {
        const waitTime = isConnectionError ? 1000 : 500;
        await new Promise(resolve => setTimeout(resolve, waitTime));
      } else {
        throw error;
      }
    }
  }
}

/**
 * 广播消息到所有 tabs
 */
export async function broadcastMessage(type, data = {}) {
  const tabs = await chrome.tabs.query({});
  const promises = tabs.map(tab =>
    chrome.tabs.sendMessage(tab.id, { type, data }).catch(() => {})
  );
  return Promise.all(promises);
}

/**
 * 发送状态更新
 */
export function sendStateUpdate(state) {
  chrome.runtime.sendMessage(
    { type: 'STATE_UPDATE', data: state },
    () => {
      if (chrome.runtime.lastError) {
        // 忽略：Popup 可能未打开
      }
    }
  );
}

/**
 * 发送进度更新
 */
export function sendProgress(progress) {
  chrome.runtime.sendMessage(
    { type: 'SCRAPING_PROGRESS', data: progress },
    () => {
      if (chrome.runtime.lastError) {
        // 忽略
      }
    }
  );
}

/**
 * 检查扩展 context 是否有效
 */
export function isContextValid() {
  try {
    return !!(chrome.runtime && chrome.runtime.id);
  } catch (error) {
    return false;
  }
}

