# 代码优化摘要

## 优化目标
保持功能不变的前提下，使代码更优雅、简洁，删除冗余内容。

## 主要优化内容

### 1. 创建共享工具模块

#### 新增文件：
- **`src/utils/api-utils.js`** - API 调用统一封装
  - `fetchWithTimeout()` - 带超时的 fetch 请求
  - `fetchJSON()` - JSON 请求封装
  - `uploadAsins()` - ASIN 上传
  - `querySellerStats()` - 商家统计查询
  - `fetchSellerDetail()` - 获取商家详情
  - `parseSellerDetail()` - 解析商家详情 HTML

- **`src/utils/storage-utils.js`** - Storage 操作统一封装
  - `getStorage()` / `setStorage()` / `removeStorage()` - 基础存储操作
  - `getCurrentUser()` - 获取当前用户
  - `checkAuth()` / `saveAuth()` / `clearAuth()` - 认证管理
  - `getApiConfig()` / `saveApiConfig()` - API 配置
  - `getPanelState()` / `savePanelState()` - 面板状态

- **`src/utils/message-utils.js`** - 消息通信统一封装
  - `sendToBackground()` - 发送到 background
  - `sendToTab()` - 发送到 tab（带重试）
  - `broadcastMessage()` - 广播消息
  - `sendStateUpdate()` / `sendProgress()` - 状态更新
  - `isContextValid()` - 检查扩展 context

### 2. 简化现有工具模块

#### `src/utils/amazon-sites.js`
**优化前问题：**
- 站点数组格式冗余，每行一个对象
- 包含未使用的 `api` 字段
- 导出了未使用的函数

**优化后：**
```javascript
// 紧凑的站点配置
export const AMAZON_SITES = [
  { id: 203, code: 'IE', name: '爱尔兰', currency: 'EUR', domain: 'amazon.ie' },
  // ...
];

// 新增域名映射对象
export const SITE_DOMAIN_MAP = AMAZON_SITES.reduce((map, site) => {
  map[site.code] = site.domain;
  return map;
}, {});

// 简化函数实现
export function getSiteByHostname(hostname) {
  if (!hostname) return null;
  const host = hostname.toLowerCase();
  return AMAZON_SITES.find(site => host.includes(site.domain)) || null;
}
```

**减少代码行数：** 219行 → 37行（减少83%）

#### `src/utils/constants.js`
**优化前问题：**
- 包含大量未使用的选择器常量
- 消息类型和状态常量未被使用
- 注释冗余

**优化后：**
```javascript
export const API_CONFIG = {
  BASE_URL: 'https://njdk.xyz:18012',
  UPLOAD_ENDPOINT: 'https://njdk.xyz:18012/api/upload-asins',
  COLLECTION_ENDPOINT: 'https://njdk.xyz:18012/api/collection/upload',
  UPDATE_CHECK_URL: 'https://zxhgs.com/extension-latest/updates.json',
  TIMEOUT: 30000,
  UPDATE_CHECK_INTERVAL: 24 * 60 * 60 * 1000
};

export const SCRAPING_CONFIG = {
  PAGE_LOAD_DELAY: 2000,
  MAX_RETRIES: 3,
  RETRY_DELAY: 1000,
  FETCH_INTERVAL: 2000
};
```

**减少代码行数：** 64行 → 16行（减少75%）

#### `src/utils/pagination-handler.js`
**优化前问题：**
- 导入了未使用的 `SELECTORS`
- `hasNextPage()` 函数有冗余的备用逻辑

**优化后：**
- 移除未使用的导入
- 简化 `hasNextPage()` 逻辑

### 3. 删除的冗余内容

#### 代码冗余
- ❌ 删除重复的 `AMAZON_SITES` 配置（在多个文件中重复）
- ❌ 删除未使用的 API 字段
- ❌ 删除未使用的导出函数（`getAllDomains`, `getAllDomainPatterns`）
- ❌ 删除未使用的常量（`MESSAGE_TYPES`, `STATUS`, `SELECTORS`）

#### 注释冗余
- ✅ 保留必要的函数说明
- ❌ 删除过于详细的实现注释
- ❌ 删除显而易见的注释

### 4. 待优化文件（大文件分批处理）

接下来将优化以下文件：
- [ ] `src/background/service-worker.js` (1252行)
- [ ] `src/content/content-script.js` (921行)
- [ ] `src/content/product-panel.js` (900行)
- [ ] `src/popup/popup.js` (2297行)
- [ ] `src/popup/login.js` (324行)

## 优化原则

1. **DRY（Don't Repeat Yourself）**
   - 提取重复的代码到工具函数
   - 统一错误处理模式
   - 统一 API 调用模式

2. **单一职责原则**
   - 每个模块只负责一个功能领域
   - API 调用、存储操作、消息通信分离

3. **保持功能不变**
   - 所有优化不改变现有功能
   - 保持 API 接口兼容
   - 保持用户体验一致

4. **提高可维护性**
   - 减少代码重复
   - 统一命名规范
   - 清晰的模块划分

## 预期收益

- **代码量减少：** 预计减少 20-30% 的总代码行数
- **可维护性提升：** 公共逻辑集中管理，修改一处即可
- **可测试性提升：** 工具函数独立，易于单元测试
- **性能优化：** 统一的错误处理和重试机制
- **代码可读性：** 去除冗余注释和重复代码

## 下一步计划

1. 优化 `service-worker.js`
   - 使用 api-utils 替换重复的 fetch 逻辑
   - 使用 storage-utils 替换存储操作
   - 使用 message-utils 替换消息发送

2. 优化 `content-script.js`
   - 删除重复的 AMAZON_SITES 配置
   - 使用共享工具模块

3. 优化 `popup.js` 和 `login.js`
   - 简化状态管理
   - 使用共享工具模块

4. 优化 `product-panel.js`
   - 简化 DOM 操作
   - 使用共享的存储和消息工具

## 注意事项

- ✅ 所有优化保持向后兼容
- ✅ 不改变用户可见的功能
- ✅ 保留所有关键日志（console.log/error）
- ✅ 保持错误提示的友好性

