/**
 * 登录页面脚本
 */

// DOM元素
const elements = {
  loginForm: document.getElementById('loginForm'),
  username: document.getElementById('username'),
  password: document.getElementById('password'),
  rememberMe: document.getElementById('rememberMe'),
  loginButton: document.getElementById('loginButton'),
  buttonText: document.getElementById('buttonText'),
  errorMessage: document.getElementById('errorMessage'),
  errorText: document.getElementById('errorText'),
  versionInfo: document.getElementById('versionInfo')
}

// API 配置
const API_CONFIG = {
  // 从扩展配置中读取，默认使用线上地址
  BASE_URL: 'https://njdk.xyz:18012'
  // BASE_URL: 'http://localhost:4500'
}

/**
 * 初始化
 */
async function init() {
  console.log('[Login] 初始化登录页面...')

  // 显示版本号
  const manifest = chrome.runtime.getManifest()
  elements.versionInfo.textContent = `v${manifest.version}`

  // 加载已保存的用户名
  loadRememberedUsername()

  // 绑定事件
  bindEvents()

  // 检查是否已登录（自动跳转）
  await checkLoginStatus()

  // 自动聚焦到第一个空输入框
  autoFocus()

  console.log('[Login] 初始化完成')
}

/**
 * 绑定事件
 */
function bindEvents() {
  // 表单提交
  elements.loginForm.addEventListener('submit', handleLogin)

  // Enter键快捷登录
  elements.password.addEventListener('keypress', (e) => {
    if (e.key === 'Enter') {
      e.preventDefault()
      handleLogin(e)
    }
  })

  // 清除错误提示
  elements.username.addEventListener('input', clearError)
  elements.password.addEventListener('input', clearError)
}

/**
 * 处理登录
 */
async function handleLogin(e) {
  e.preventDefault()
  console.log('[Login] 开始登录...')

  const username = elements.username.value.trim()
  const password = elements.password.value.trim()

  // 验证输入
  if (!username) {
    showError('请输入账号')
    elements.username.focus()
    return
  }

  if (!password) {
    showError('请输入密码')
    elements.password.focus()
    return
  }

  if (username.length < 3) {
    showError('账号长度不能少于3位')
    elements.username.focus()
    return
  }

  if (password.length < 6) {
    showError('密码长度不能少于6位')
    elements.password.focus()
    return
  }

  try {
    // 显示加载状态
    setLoading(true)
    clearError()

    // 调用登录API
    const result = await loginApi(username, password)

    if (result.success) {
      console.log('[Login] 登录成功:', result.data)

      // 保存登录信息到storage
      await chrome.storage.local.set({
        isAuthenticated: true,
        currentUser: {
          username: username,
          userId: result.data.userId,
          isAdmin: result.data.isAdmin,
          status: result.data.status
        },
        loginTime: Date.now()
      })

      // 如果选择了记住账号，保存用户名
      if (elements.rememberMe.checked) {
        await chrome.storage.local.set({
          rememberedUsername: username
        })
      } else {
        await chrome.storage.local.remove('rememberedUsername')
      }

      // 显示成功消息
      elements.buttonText.textContent = '✓ 登录成功'
      elements.loginButton.style.background = 'linear-gradient(135deg, #10b981, #059669)'

      // 延迟跳转，让用户看到成功提示
      setTimeout(() => {
        // 跳转到主页面
        window.location.href = 'popup.html'
      }, 500)
    } else {
      console.warn('[Login] 登录失败:', result.message)
      showError(result.message || '账号或密码错误，请重试')

      // 聚焦到密码框
      elements.password.focus()
      elements.password.select()
    }
  } catch (error) {
    console.error('[Login] 登录异常:', error)
    showError('登录服务暂时不可用，请稍后重试')
  } finally {
    setLoading(false)
  }
}

/**
 * 调用登录API
 */
async function loginApi(username, password) {
  try {
    // 从storage获取API配置
    const { apiEndpoint } = await chrome.storage.local.get('apiEndpoint')
    let baseUrl = API_CONFIG.BASE_URL

    // 如果有配置的API地址，从中提取基础URL
    if (apiEndpoint) {
      try {
        const url = new URL(apiEndpoint)
        baseUrl = url.origin
      } catch (e) {
        console.warn('[Login] 无法解析API地址，使用默认配置')
      }
    }

    const loginUrl = `${baseUrl}/api/users/login`
    console.log('[Login] 请求登录API:', loginUrl)

    const controller = new AbortController()
    const timeoutId = setTimeout(() => controller.abort(), 10000) // 10秒超时

    const response = await fetch(loginUrl, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json'
      },
      body: JSON.stringify({
        username: username,
        password: password
      }),
      signal: controller.signal
    })

    clearTimeout(timeoutId)

    const result = await response.json()
    console.log('[Login] API响应:', result)

    return result
  } catch (error) {
    console.error('[Login] API请求失败:', error)

    if (error.name === 'AbortError') {
      return {
        success: false,
        message: '请求超时，请检查网络连接'
      }
    }

    return {
      success: false,
      message: '无法连接到服务器，请检查网络或API配置'
    }
  }
}

/**
 * 检查登录状态
 */
async function checkLoginStatus() {
  try {
    const { isAuthenticated, loginTime, currentUser } = await chrome.storage.local.get([
      'isAuthenticated',
      'loginTime',
      'currentUser'
    ])

    if (isAuthenticated && currentUser) {
      // 检查登录是否过期（24小时）
      const now = Date.now()
      const expired = loginTime && now - loginTime > 24 * 60 * 60 * 1000

      if (!expired) {
        console.log('[Login] 已登录，跳转到主页面')
        // 已登录且未过期，直接跳转
        window.location.href = 'popup.html'
        return true
      } else {
        console.log('[Login] 登录已过期，需要重新登录')
        // 清除过期的登录信息
        await chrome.storage.local.remove(['isAuthenticated', 'currentUser', 'loginTime'])
      }
    }
  } catch (error) {
    console.error('[Login] 检查登录状态失败:', error)
  }

  return false
}

/**
 * 加载记住的用户名
 */
async function loadRememberedUsername() {
  try {
    const { rememberedUsername } = await chrome.storage.local.get('rememberedUsername')

    if (rememberedUsername) {
      elements.username.value = rememberedUsername
      elements.rememberMe.checked = true
    }
  } catch (error) {
    console.error('[Login] 加载记住的用户名失败:', error)
  }
}

/**
 * 设置加载状态
 */
function setLoading(isLoading) {
  elements.loginButton.disabled = isLoading
  elements.loginButton.classList.toggle('loading', isLoading)

  if (isLoading) {
    elements.buttonText.textContent = ''
    elements.username.disabled = true
    elements.password.disabled = true
    elements.rememberMe.disabled = true
  } else {
    elements.buttonText.textContent = '立即登录'
    elements.username.disabled = false
    elements.password.disabled = false
    elements.rememberMe.disabled = false
  }
}

/**
 * 显示错误消息
 */
function showError(message) {
  elements.errorText.textContent = message
  elements.errorMessage.style.display = 'flex'
}

/**
 * 清除错误消息
 */
function clearError() {
  elements.errorMessage.style.display = 'none'
  elements.errorText.textContent = ''
}

/**
 * 自动聚焦
 */
function autoFocus() {
  if (!elements.username.value) {
    elements.username.focus()
  } else {
    elements.password.focus()
  }
}

// 初始化
document.addEventListener('DOMContentLoaded', init)

console.log('[Login] 登录脚本已加载')

