/**
 * Popup脚本
 * 处理用户交互和状态显示
 */

// 亚马逊站点配置
const AMAZON_SITES = [
  { id: 203, code: 'IE', name: '爱尔兰', currency: 'EUR', domain: 'amazon.ie', flag: '🇮🇪' },
  { id: 1, code: 'US', name: '美国', currency: 'USD', domain: 'amazon.com', flag: '🇺🇸' },
  { id: 2, code: 'UK', name: '英国', currency: 'GBP', domain: 'amazon.co.uk', flag: '🇬🇧' },
  { id: 3, code: 'DE', name: '德国', currency: 'EUR', domain: 'amazon.de', flag: '🇩🇪' },
  { id: 4, code: 'FR', name: '法国', currency: 'EUR', domain: 'amazon.fr', flag: '🇫🇷' },
  { id: 5, code: 'IT', name: '意大利', currency: 'EUR', domain: 'amazon.it', flag: '🇮🇹' },
  { id: 6, code: 'ES', name: '西班牙', currency: 'EUR', domain: 'amazon.es', flag: '🇪🇸' },
  { id: 7, code: 'JP', name: '日本', currency: 'JPY', domain: 'amazon.co.jp', flag: '🇯🇵' },
  { id: 8, code: 'CA', name: '加拿大', currency: 'CAD', domain: 'amazon.ca', flag: '🇨🇦' },
  { id: 9, code: 'AU', name: '澳大利亚', currency: 'AUD', domain: 'amazon.com.au', flag: '🇦🇺' },
  { id: 10, code: 'MX', name: '墨西哥', currency: 'MXN', domain: 'amazon.com.mx', flag: '🇲🇽' },
  { id: 11, code: 'BR', name: '巴西', currency: 'BRL', domain: 'amazon.com.br', flag: '🇧🇷' },
  { id: 12, code: 'IN', name: '印度', currency: 'INR', domain: 'amazon.in', flag: '🇮🇳' },
  { id: 13, code: 'SG', name: '新加坡', currency: 'SGD', domain: 'amazon.sg', flag: '🇸🇬' },
  { id: 14, code: 'AE', name: '阿联酋', currency: 'AED', domain: 'amazon.ae', flag: '🇦🇪' },
  { id: 15, code: 'SA', name: '沙特阿拉伯', currency: 'SAR', domain: 'amazon.sa', flag: '🇸🇦' },
  { id: 121, code: 'NL', name: '荷兰', currency: 'EUR', domain: 'amazon.nl', flag: '🇳🇱' },
  { id: 122, code: 'PL', name: '波兰', currency: 'PLN', domain: 'amazon.pl', flag: '🇵🇱' },
  { id: 123, code: 'SE', name: '瑞典', currency: 'SEK', domain: 'amazon.se', flag: '🇸🇪' },
  { id: 124, code: 'BE', name: '比利时', currency: 'EUR', domain: 'amazon.com.be', flag: '🇧🇪' }
]

/**
 * 检查URL是否为亚马逊站点
 */
function isAmazonUrl(url) {
  if (!url) return false

  for (const site of AMAZON_SITES) {
    if (url.includes(site.domain)) {
      return true
    }
  }

  return false
}

// DOM元素
const elements = {
  sitesGrid: document.getElementById('sitesGrid'),
  siteSearch: document.getElementById('siteSearch'),
  apiEndpoint: document.getElementById('apiEndpoint'),
  saveConfig: document.getElementById('saveConfig'),
  startButton: document.getElementById('startButton'),
  pauseButton: document.getElementById('pauseButton'),
  resumeButton: document.getElementById('resumeButton'),
  cancelButton: document.getElementById('cancelButton'),
  resetButton: document.getElementById('resetButton'),
  pageUrl: document.getElementById('pageUrl'),
  statusCard: document.getElementById('statusCard'),
  statusDot: document.getElementById('statusDot'),
  statusText: document.getElementById('statusText'),
  currentPage: document.getElementById('currentPage'),
  totalPages: document.getElementById('totalPages'),
  asinCount: document.getElementById('asinCount'),
  progressBarContainer: document.getElementById('progressBarContainer'),
  progressBar: document.getElementById('progressBar'),
  errorMessage: document.getElementById('errorMessage'),
  successMessage: document.getElementById('successMessage'),
  // 用户信息相关元素
  userInfo: document.getElementById('userInfo'),
  currentUsername: document.getElementById('currentUsername'),
  dashboardButton: document.getElementById('dashboardButton'),
  logoutButton: document.getElementById('logoutButton'),
  // 商家统计相关元素
  querySellerStats: document.getElementById('querySellerStats'),
  minAsinCount: document.getElementById('minAsinCount'),
  minOfferCount: document.getElementById('minOfferCount'),
  statsLoading: document.getElementById('statsLoading'),
  statsResults: document.getElementById('statsResults'),
  statsError: document.getElementById('statsError'),
  statsCount: document.getElementById('statsCount'),
  statsTableBody: document.getElementById('statsTableBody'),
  // 批量操作相关元素
  batchActions: document.getElementById('batchActions'),
  selectAll: document.getElementById('selectAll'),
  headerCheckbox: document.getElementById('headerCheckbox'),
  batchGetDetails: document.getElementById('batchGetDetails'),
  exportSellerData: document.getElementById('exportSellerData'),
  exportStatsData: document.getElementById('exportStatsData'),
  batchInfo: document.getElementById('batchInfo'),
  batchProgress: document.getElementById('batchProgress'),
  batchProgressText: document.getElementById('batchProgressText'),
  batchProgressFill: document.getElementById('batchProgressFill'),
  // 商家统计数据缓存
  sellerStatsData: null,
  // 商家详情相关元素
  sellerDetailModal: document.getElementById('sellerDetailModal'),
  sellerDetailContent: document.getElementById('sellerDetailContent'),
  closeSellerDetail: document.getElementById('closeSellerDetail'),
  // 产品面板按钮
  toggleProductPanel: document.getElementById('toggleProductPanel')
}

// 当前状态
let currentState = {
  status: 'idle',
  currentPage: 0,
  totalPages: 0,
  asinCount: 0,
  error: null
}

/**
 * 获取当前激活站点
 */
async function getCurrentSite() {
  try {
    const [tab] = await chrome.tabs.query({ active: true, currentWindow: true })
    if (!tab || !tab.url) return null

    for (const site of AMAZON_SITES) {
      if (tab.url.includes(site.domain)) {
        return site.code
      }
    }
  } catch (error) {
    console.error('[Popup] 获取当前站点失败:', error)
  }
  return null
}

/**
 * 渲染站点按钮
 */
async function renderSiteButtons(filter = '') {
  const currentSite = await getCurrentSite()
  const filterLower = filter.toLowerCase()

  // 过滤站点
  const filteredSites = AMAZON_SITES.filter(site => {
    if (!filter) return true
    return (
      site.code.toLowerCase().includes(filterLower) ||
      site.name.toLowerCase().includes(filterLower) ||
      site.domain.toLowerCase().includes(filterLower)
    )
  })

  // 清空并生成按钮
  elements.sitesGrid.innerHTML = ''

  filteredSites.forEach(site => {
    const button = document.createElement('button')
    button.className = 'site-btn'
    if (site.code === currentSite) {
      button.classList.add('active')
    }

    button.innerHTML = `
      <div class="site-btn-flag">${site.flag}</div>
      <div class="site-btn-code">${site.code}</div>
      <div class="site-btn-name">${site.name}</div>
    `

    button.title = `打开 ${site.name} (${site.domain})`

    button.addEventListener('click', () => {
      openAmazonSite(site)
    })

    elements.sitesGrid.appendChild(button)
  })

  // 如果没有结果
  if (filteredSites.length === 0) {
    elements.sitesGrid.innerHTML = '<div style="grid-column: 1/-1; text-align: center; padding: 20px; color: #999;">未找到匹配的站点</div>'
  }
}

/**
 * 打开亚马逊站点
 */
function openAmazonSite(site) {
  const openBestsellers = document.getElementById('openBestsellers').checked
  let url = `https://${site.domain}`

  // 如果勾选了 Bestsellers 选项，添加路径
  if (openBestsellers) {
    url += '/-/en/gp/bestsellers/?ref_=nav_cs_bestsellers'
  }

  chrome.tabs.create({ url, active: true })
  console.log('[Popup] 打开站点:', site.name, url, openBestsellers ? '(Bestsellers)' : '(首页)')
}

/**
 * 检查登录状态
 */
async function checkLoginStatus() {
  try {
    const { isAuthenticated, loginTime, currentUser } = await chrome.storage.local.get([
      'isAuthenticated',
      'loginTime',
      'currentUser'
    ])

    if (!isAuthenticated || !currentUser) {
      console.log('[Popup] 未登录，重定向到登录页面')
      window.location.href = 'login.html'
      return false
    }

    // 检查登录是否过期（24小时）
    const now = Date.now()
    const expired = loginTime && now - loginTime > 24 * 60 * 60 * 1000

    if (expired) {
      console.log('[Popup] 登录已过期，重定向到登录页面')
      // 清除过期的登录信息
      await chrome.storage.local.remove(['isAuthenticated', 'currentUser', 'loginTime'])
      window.location.href = 'login.html'
      return false
    }

    console.log('[Popup] 已登录:', currentUser.username)

    // 显示用户信息
    displayUserInfo(currentUser)

    return true
  } catch (error) {
    console.error('[Popup] 检查登录状态失败:', error)
    window.location.href = 'login.html'
    return false
  }
}

/**
 * 显示用户信息
 */
function displayUserInfo(currentUser) {
  if (elements.currentUsername && currentUser && currentUser.username) {
    elements.currentUsername.textContent = currentUser.username
    elements.currentUsername.title = `当前用户: ${currentUser.username}`
  }
}

/**
 * 打开采集统计面板
 */
function openDashboard() {
  try {
    // 从配置中获取API地址，提取基础URL
    chrome.storage.local.get('apiEndpoint', (result) => {
      let dashboardUrl = 'https://njdk.xyz:18012/'

      // 如果有配置的API地址，尝试提取基础URL
      if (result.apiEndpoint) {
        try {
          const url = new URL(result.apiEndpoint)
          dashboardUrl = url.origin + '/'
        } catch (e) {
          console.warn('[Popup] 无法解析API地址，使用默认URL')
        }
      }

      console.log('[Popup] 打开采集统计面板:', dashboardUrl)

      // 在新标签页打开统计面板
      chrome.tabs.create({ url: dashboardUrl, active: true })
    })
  } catch (error) {
    console.error('[Popup] 打开统计面板失败:', error)
  }
}

/**
 * 退出登录
 */
async function handleLogout() {
  try {
    const confirmed = confirm('确定要退出登录吗？')
    if (!confirmed) return

    // 清除登录信息
    await chrome.storage.local.remove(['isAuthenticated', 'currentUser', 'loginTime'])

    console.log('[Popup] 已退出登录')

    // 跳转到登录页面
    window.location.href = 'login.html'
  } catch (error) {
    console.error('[Popup] 退出登录失败:', error)
    alert('退出登录失败，请重试')
  }
}

/**
 * 初始化
 */
async function init() {
  console.log('[Popup] 初始化...')

  // 首先检查登录状态
  const isLoggedIn = await checkLoginStatus()
  if (!isLoggedIn) {
    return // 如果未登录，已经重定向到登录页面
  }

  // 显示当前版本号（头部和底部）
  const manifest = chrome.runtime.getManifest()

  // 头部版本徽章
  const versionBadge = document.getElementById('versionBadge')
  if (versionBadge) {
    versionBadge.textContent = `v${manifest.version}`
  }

  // 底部版本文本
  const versionEl = document.getElementById('footerVersion')
  if (versionEl) {
    versionEl.textContent = `Version ${manifest.version}`
  }

  // 渲染站点按钮
  await renderSiteButtons()

  // 加载保存的API配置
  const { apiEndpoint, scrapingMode, concurrency } = await chrome.storage.local.get(['apiEndpoint', 'scrapingMode', 'concurrency'])
  if (apiEndpoint) {
    elements.apiEndpoint.value = apiEndpoint
  } else {
    elements.apiEndpoint.value = 'https://njdk.xyz:18012/api/upload-asins'
    // elements.apiEndpoint.value = 'http://localhost:4500/api/upload-asins'
  }

  // 加载采集模式设置（默认单页）
  const mode = scrapingMode || 'single'
  const modeRadio = document.querySelector(`input[name="scrapingMode"][value="${mode}"]`)
  if (modeRadio) {
    modeRadio.checked = true
  }

  // 加载并发数设置（默认5）
  const concurrencySelect = document.getElementById('concurrency')
  if (concurrencySelect) {
    concurrencySelect.value = (concurrency || 5).toString()
  }

  // 加载当前状态
  await loadCurrentState()

  // 加载页面信息
  await loadPageInfo()

  // 同步产品面板按钮状态
  await syncPanelButtonState()

  // 检查是否有更新可用
  await checkForUpdate()

  // 绑定事件
  bindEvents()

  // 恢复缓存的查询结果（在API配置加载后）
  await restoreCachedQueryResult()

  console.log('[Popup] 初始化完成')
}

/**
 * 检查更新
 */
async function checkForUpdate() {
  try {
    const { updateAvailable, updateInfo } = await chrome.storage.local.get([
      'updateAvailable',
      'updateInfo'
    ])

    if (updateAvailable && updateInfo) {
      showUpdateBanner(updateInfo)
    }
  } catch (error) {
    console.error('[Popup] 检查更新失败:', error)
  }
}

/**
 * 显示更新横幅
 */
function showUpdateBanner(updateInfo) {
  const banner = document.getElementById('updateBanner')
  const versionSpan = document.getElementById('updateVersion')

  if (banner && versionSpan) {
    versionSpan.textContent = `v${updateInfo.latest_version} 可用`
    banner.style.display = 'block'
  }
}

/**
 * 手动检查更新
 */
async function manualCheckUpdate() {
  try {
    const response = await chrome.runtime.sendMessage({ type: 'CHECK_UPDATE' })

    if (response.success) {
      if (response.hasUpdate && response.updateInfo) {
        showUpdateBanner(response.updateInfo)
        alert(`发现新版本 v${response.updateInfo.latest_version}！\n\n${response.updateInfo.release_notes}`)
      } else {
        alert('当前已是最新版本！')
      }
    } else {
      alert(`检查更新失败：${response.error || '未知错误'}`)
    }
  } catch (error) {
    console.error('[Popup] 手动检查更新失败:', error)
    alert(`检查更新失败：${error.message}`)
  }
}

/**
 * 切换产品面板显示
 */
async function toggleProductPanel() {
  try {
    console.log('[Popup] 开始切换产品面板')
    const [tab] = await chrome.tabs.query({ active: true, currentWindow: true })

    if (!tab || !tab.url || !isAmazonUrl(tab.url)) {
      alert('请在亚马逊页面使用此功能')
      return
    }

    // 先获取当前状态来预测切换后的状态
    const btnText = elements.toggleProductPanel?.querySelector('.toggle-text')
    const currentText = btnText?.textContent || '显示'
    const willBeVisible = currentText === '显示'

    console.log('[Popup] 当前按钮文字:', currentText, '切换后将显示面板:', willBeVisible)

    // 立即更新按钮状态（乐观更新）
    updatePanelButton(willBeVisible)

    console.log('[Popup] 发送切换消息到 content script')

    try {
      // 发送消息到content script
      const response = await chrome.tabs.sendMessage(tab.id, { type: 'TOGGLE_PRODUCT_PANEL' })

      console.log('[Popup] 收到响应:', response)

      // 根据返回的状态再次更新按钮（确保状态同步）
      if (response && response.success) {
        updatePanelButton(response.isVisible)
      } else {
        console.error('[Popup] 响应失败或不成功，回滚按钮状态')
        // 如果失败，回滚按钮状态
        updatePanelButton(!willBeVisible)
      }
    } catch (error) {
      console.error('[Popup] 发送消息失败，回滚按钮状态:', error)
      // 如果发送消息失败，回滚按钮状态
      updatePanelButton(!willBeVisible)
      throw error
    }

  } catch (error) {
    console.error('[Popup] 切换产品面板失败:', error)
    alert('操作失败，请刷新页面后重试')
  }
}

/**
 * 更新产品面板按钮状态
 */
function updatePanelButton(isVisible) {
  console.log('[Popup] 更新按钮状态:', isVisible, '类型:', typeof isVisible)

  if (!elements.toggleProductPanel) {
    console.error('[Popup] 按钮元素不存在')
    return
  }

  const btnText = elements.toggleProductPanel.querySelector('.toggle-text')
  const btnIcon = elements.toggleProductPanel.querySelector('.toggle-icon')

  console.log('[Popup] 按钮元素:', {
    btnText: btnText,
    btnIcon: btnIcon,
    currentText: btnText?.textContent,
    currentIcon: btnIcon?.textContent
  })

  if (!btnText || !btnIcon) {
    console.error('[Popup] 按钮文本或图标元素不存在')
    return
  }

  if (isVisible) {
    console.log('[Popup] 设置为"隐藏"状态...')
    btnText.textContent = '隐藏'
    btnIcon.textContent = '👁️‍🗨️'
    elements.toggleProductPanel.style.background = 'linear-gradient(135deg, #64748b 0%, #475569 100%)'
    console.log('[Popup] 按钮已设置为"隐藏"状态，验证:', {
      text: btnText.textContent,
      icon: btnIcon.textContent
    })
  } else {
    console.log('[Popup] 设置为"显示"状态...')
    btnText.textContent = '显示'
    btnIcon.textContent = '👁️'
    elements.toggleProductPanel.style.background = ''
    console.log('[Popup] 按钮已设置为"显示"状态，验证:', {
      text: btnText.textContent,
      icon: btnIcon.textContent
    })
  }
}

/**
 * 同步产品面板按钮状态
 */
async function syncPanelButtonState() {
  try {
    const [tab] = await chrome.tabs.query({ active: true, currentWindow: true })

    if (!tab || !tab.url || !isAmazonUrl(tab.url)) {
      return
    }

    // 获取当前面板状态
    const response = await chrome.tabs.sendMessage(tab.id, { type: 'GET_PANEL_STATE' })

    if (response && response.success) {
      updatePanelButton(response.isVisible)
    }
  } catch (error) {
    // 页面可能还未加载完成，忽略错误
    console.log('[Popup] 同步面板状态失败:', error.message)
  }
}

/**
 * 加载页面信息（仅用于显示页面类型）
 */
async function loadPageInfo() {
  try {
    const [tab] = await chrome.tabs.query({ active: true, currentWindow: true })

    if (!tab || !tab.url) {
      elements.pageUrl.textContent = '无法获取页面信息'
      return
    }

    // 显示页面类型
    if (isAmazonUrl(tab.url)) {
      // 请求页面详细信息以获取站点信息
      try {
        const response = await sendMessageWithRetry(tab.id, { type: 'GET_PAGE_INFO' }, 2)
        if (response && response.success) {
          const info = response.data

          // 只显示站点信息
          if (info.siteName) {
            elements.pageUrl.textContent = `亚马逊 ${info.siteName} (${info.site})`
          } else if (info.site) {
            elements.pageUrl.textContent = `亚马逊 ${info.site} 站`
          } else {
            elements.pageUrl.textContent = '亚马逊页面'
          }
        } else {
          elements.pageUrl.textContent = '亚马逊页面'
        }
      } catch (error) {
        console.log('[Popup] 无法获取页面详细信息:', error.message)
        elements.pageUrl.textContent = '亚马逊页面'
      }
    } else {
      elements.pageUrl.textContent = '非亚马逊页面'
    }
  } catch (error) {
    console.error('[Popup] 加载页面信息失败:', error)
    elements.pageUrl.textContent = '页面信息加载失败'
  }
}

/**
 * 加载当前状态
 */
async function loadCurrentState() {
  try {
    const response = await chrome.runtime.sendMessage({ type: 'GET_STATUS' })
    if (response.success && response.state) {
      currentState = response.state
      updateUI()
    }
  } catch (error) {
    console.error('[Popup] 加载状态失败:', error)
  }
}

/**
 * 绑定事件
 */
function bindEvents() {
  // 站点搜索
  elements.siteSearch.addEventListener('input', (e) => {
    renderSiteButtons(e.target.value)
  })

  // 采集模式切换 - 显示/隐藏并发数设置
  const modeRadios = document.querySelectorAll('input[name="scrapingMode"]')
  modeRadios.forEach(radio => {
    radio.addEventListener('change', (e) => {
      const concurrencyGroup = document.getElementById('concurrencyGroup')
      if (concurrencyGroup) {
        // 只有后台模式才显示并发数设置
        concurrencyGroup.style.display = e.target.value === 'background' ? 'block' : 'none'
      }
    })
  })

  // 初始化并发数显示状态
  const checkedMode = document.querySelector('input[name="scrapingMode"]:checked')
  const concurrencyGroup = document.getElementById('concurrencyGroup')
  if (concurrencyGroup && checkedMode) {
    concurrencyGroup.style.display = checkedMode.value === 'background' ? 'block' : 'none'
  }

  // 保存配置
  elements.saveConfig.addEventListener('click', saveConfiguration)

  // 开始采集
  elements.startButton.addEventListener('click', startScraping)

  // 暂停采集
  elements.pauseButton.addEventListener('click', pauseScraping)

  // 继续采集
  elements.resumeButton.addEventListener('click', resumeScraping)

  // 取消采集
  elements.cancelButton.addEventListener('click', cancelScraping)

  // 重置
  elements.resetButton.addEventListener('click', resetStatus)

  // 商家统计查询
  elements.querySellerStats.addEventListener('click', querySellerStatistics)

  // 监听查询条件输入框变化，条件改变时清除不匹配的缓存显示
  elements.minAsinCount.addEventListener('input', checkAndClearCacheIfNeeded)
  elements.minOfferCount.addEventListener('input', checkAndClearCacheIfNeeded)
  elements.apiEndpoint.addEventListener('input', checkAndClearCacheIfNeeded)

  // 批量操作
  elements.selectAll.addEventListener('change', handleSelectAll)
  elements.headerCheckbox.addEventListener('change', handleSelectAll)
  elements.batchGetDetails.addEventListener('click', handleBatchGetDetails)
  elements.exportSellerData.addEventListener('click', showExportDetailPage)
  elements.exportStatsData.addEventListener('click', exportStatsData)

  // 导出详情页面
  elements.closeExportDetail = document.getElementById('closeExportDetail')
  elements.exportDetailModal = document.getElementById('exportDetailModal')
  elements.exportDetailContent = document.getElementById('exportDetailContent')
  elements.exportListContainer = document.getElementById('exportListContainer')
  elements.exportSearchInput = document.getElementById('exportSearchInput')
  elements.exportSiteFilter = document.getElementById('exportSiteFilter')
  elements.exportToJson = document.getElementById('exportToJson')
  elements.exportToCsv = document.getElementById('exportToCsv')
  elements.exportTotalCount = document.getElementById('exportTotalCount')
  elements.exportDetailCount = document.getElementById('exportDetailCount')
  elements.exportLastUpdate = document.getElementById('exportLastUpdate')

  // 绑定导出详情页面事件
  elements.closeExportDetail.addEventListener('click', () => {
    elements.exportDetailModal.style.display = 'none'
    elements.exportDetailModal.classList.remove('show')
  })

  elements.exportDetailModal.addEventListener('click', (e) => {
    if (e.target === elements.exportDetailModal) {
      elements.exportDetailModal.style.display = 'none'
      elements.exportDetailModal.classList.remove('show')
    }
  })

  elements.exportSearchInput.addEventListener('input', filterExportList)
  elements.exportSiteFilter.addEventListener('change', filterExportList)
  elements.exportToJson.addEventListener('click', () => handleExportData('json'))
  elements.exportToCsv.addEventListener('click', () => handleExportData('csv'))

  // 商家详情对话框
  elements.closeSellerDetail.addEventListener('click', () => {
    elements.sellerDetailModal.style.display = 'none'
    elements.sellerDetailModal.classList.remove('show')
  })

  // 点击模态框外部关闭
  elements.sellerDetailModal.addEventListener('click', (e) => {
    if (e.target === elements.sellerDetailModal) {
      elements.sellerDetailModal.style.display = 'none'
      elements.sellerDetailModal.classList.remove('show')
    }
  })

  // 采集统计面板按钮
  if (elements.dashboardButton) {
    elements.dashboardButton.addEventListener('click', openDashboard)
  }

  // 退出登录按钮
  if (elements.logoutButton) {
    elements.logoutButton.addEventListener('click', handleLogout)
  }

  // 更新提示相关事件
  const downloadUpdateBtn = document.getElementById('downloadUpdate')
  const closeUpdateBannerBtn = document.getElementById('closeUpdateBanner')
  const checkUpdateLink = document.getElementById('checkUpdateLink')

  if (downloadUpdateBtn) {
    downloadUpdateBtn.addEventListener('click', async () => {
      const { updateInfo } = await chrome.storage.local.get('updateInfo')
      if (updateInfo && updateInfo.download_url) {
        chrome.tabs.create({ url: updateInfo.download_url })
      }
    })
  }

  if (closeUpdateBannerBtn) {
    closeUpdateBannerBtn.addEventListener('click', () => {
      document.getElementById('updateBanner').style.display = 'none'
    })
  }

  if (checkUpdateLink) {
    checkUpdateLink.addEventListener('click', (e) => {
      e.preventDefault()
      manualCheckUpdate()
    })
  }

  // 产品面板按钮
  if (elements.toggleProductPanel) {
    elements.toggleProductPanel.addEventListener('click', toggleProductPanel)
  }

  // 监听来自background的状态更新
  chrome.runtime.onMessage.addListener(message => {
    console.log('[Popup] 收到消息:', message.type)

    switch (message.type) {
      case 'STATE_UPDATE':
        currentState = message.data
        updateUI()
        break

      case 'SCRAPING_PROGRESS':
        currentState.status = message.data.status || 'scraping'
        currentState.currentPage = message.data.currentPage
        currentState.totalPages = message.data.totalPages
        currentState.asinCount = message.data.asinCount
        updateUI()
        break

      case 'SCRAPING_CANCELLED':
        showSuccessMessage(`采集已取消，已采集 ${message.data.totalCount} 个ASIN`)
        currentState.status = 'cancelled'
        updateUI()
        break

      case 'UPLOAD_COMPLETE':
        showSuccessMessage(`成功上传 ${message.data.asinCount} 个ASIN到后端！`)
        break

      case 'UPLOAD_ERROR':
        showErrorMessage(`上传失败: ${message.data.error}`)
        break
    }
  })
}

/**
 * 保存配置
 */
async function saveConfiguration() {
  const endpoint = elements.apiEndpoint.value.trim()
  const modeRadio = document.querySelector('input[name="scrapingMode"]:checked')
  const scrapingMode = modeRadio ? modeRadio.value : 'single'
  const concurrencySelect = document.getElementById('concurrency')
  const concurrency = concurrencySelect ? parseInt(concurrencySelect.value) : 5

  if (!endpoint) {
    alert('请输入API地址')
    return
  }

  try {
    // 检查API端点是否改变
    const cached = await getCachedQueryParams()
    const endpointChanged = cached && cached.queryParams && cached.queryParams.apiEndpoint !== endpoint

    // 保存配置
    await chrome.storage.local.set({
      apiEndpoint: endpoint,
      scrapingMode: scrapingMode,
      concurrency: concurrency
    })

    await chrome.runtime.sendMessage({
      type: 'UPDATE_CONFIG',
      data: { endpoint, scrapingMode, concurrency }
    })

    // 如果API端点改变，清除缓存的显示
    if (endpointChanged) {
      await checkAndClearCacheIfNeeded()
    }

    // 显示保存成功提示
    const originalText = elements.saveConfig.textContent
    elements.saveConfig.textContent = '✓ 已保存'
    elements.saveConfig.style.background = '#28a745'

    setTimeout(() => {
      elements.saveConfig.textContent = originalText
      elements.saveConfig.style.background = ''
    }, 2000)

    console.log('[Popup] 配置已保存')
  } catch (error) {
    console.error('[Popup] 保存配置失败:', error)
    alert('保存配置失败，请重试')
  }
}

/**
 * 等待并发送消息到content script（带重试）
 */
async function sendMessageWithRetry(tabId, message, maxRetries = 3) {
  for (let i = 0; i < maxRetries; i++) {
    try {
      const response = await chrome.tabs.sendMessage(tabId, message)
      return response
    } catch (error) {
      // 检查是否是连接错误
      const isConnectionError = error.message.includes('Receiving end does not exist') ||
        error.message.includes('Could not establish connection')

      console.log(`[Popup] 尝试 ${i + 1}/${maxRetries} 失败:`, error.message)

      if (i < maxRetries - 1) {
        // 如果是连接错误，可能需要稍长一点的等待
        const waitTime = isConnectionError ? 1000 : 500
        await new Promise(resolve => setTimeout(resolve, waitTime))
      } else {
        throw error
      }
    }
  }
}

/**
 * 开始采集
 */
async function startScraping() {
  console.log('[Popup] 开始采集...')

  // 检查是否在正确的页面
  const [tab] = await chrome.tabs.query({ active: true, currentWindow: true })

  if (!tab || !tab.url) {
    alert('无法获取当前页面信息')
    return
  }

  if (!isAmazonUrl(tab.url)) {
    alert('请在亚马逊（Amazon）页面使用此功能！')
    return
  }

  // 检查API配置
  const endpoint = elements.apiEndpoint.value.trim()
  if (!endpoint) {
    alert('请先配置API地址！')
    return
  }

  // 获取页面信息并确认
  try {
    const response = await sendMessageWithRetry(tab.id, { type: 'GET_PAGE_INFO' })
    if (response && response.success) {
      const info = response.data
      let confirmMsg = '准备开始采集ASIN\n\n'

      if (info.site) {
        confirmMsg += `🌍 站点: ${info.site}\n`
      }
      if (info.category) {
        confirmMsg += `📁 分类: ${info.category}\n`
      }
      if (info.searchKeyword) {
        confirmMsg += `🔍 搜索: "${info.searchKeyword}"\n`
      }
      if (info.totalResults) {
        confirmMsg += `📦 约 ${info.totalResults} 个商品\n`
      }

      confirmMsg += '\n确定要开始采集吗？'

      if (!confirm(confirmMsg)) {
        return
      }
    }
  } catch (error) {
    console.warn('[Popup] 无法获取页面信息:', error.message)
    // 简单确认
    if (!confirm('无法获取页面详细信息\n\n是否继续采集？')) {
      return
    }
  }

  try {
    // 重置状态
    currentState = {
      status: 'scraping',
      currentPage: 0,
      totalPages: 0,
      asinCount: 0,
      error: null
    }
    updateUI()

    // 发送消息到content script（带重试）
    await sendMessageWithRetry(tab.id, { type: 'START_SCRAPING' })

    console.log('[Popup] 采集任务已启动')
  } catch (error) {
    console.error('[Popup] 启动采集失败:', error)
    showErrorMessage(`启动失败: ${error.message}\n\n请刷新页面后重试`)
    currentState.status = 'error'
    updateUI()
  }
}

/**
 * 暂停采集
 */
async function pauseScraping() {
  try {
    const [tab] = await chrome.tabs.query({ active: true, currentWindow: true })
    await chrome.tabs.sendMessage(tab.id, { type: 'PAUSE_SCRAPING' })

    currentState.status = 'paused'
    updateUI()

    console.log('[Popup] 已发送暂停指令')
  } catch (error) {
    console.error('[Popup] 暂停失败:', error)
    showErrorMessage(`暂停失败: ${error.message}`)
  }
}

/**
 * 继续采集
 */
async function resumeScraping() {
  try {
    const [tab] = await chrome.tabs.query({ active: true, currentWindow: true })
    await chrome.tabs.sendMessage(tab.id, { type: 'RESUME_SCRAPING' })

    currentState.status = 'scraping'
    updateUI()

    console.log('[Popup] 已发送继续指令')
  } catch (error) {
    console.error('[Popup] 继续失败:', error)
    showErrorMessage(`继续失败: ${error.message}`)
  }
}

/**
 * 取消采集
 */
async function cancelScraping() {
  try {
    const confirmed = confirm('确定要取消当前采集任务吗？已采集的数据将会保留。')
    if (!confirmed) return

    const [tab] = await chrome.tabs.query({ active: true, currentWindow: true })

    try {
      const response = await chrome.tabs.sendMessage(tab.id, { type: 'CANCEL_SCRAPING' })
      console.log('[Popup] 已发送取消指令', response)

      // 如果content script返回失败（例如没有正在进行的任务），我们也应该重置状态
      if (response && !response.success) {
        console.warn('[Popup] Content script报告取消失败:', response.error)
        // 强制重置状态，防止UI卡死
        await resetStatus()
      }
    } catch (error) {
      // 如果content script不存在或无法连接，直接重置状态
      console.warn('[Popup] 无法发送取消消息，直接重置状态:', error.message)
      await resetStatus()
    }
  } catch (error) {
    console.error('[Popup] 取消失败:', error)
    showErrorMessage(`取消失败: ${error.message}`)
    // 发生严重错误时也尝试重置
    await resetStatus()
  }
}

/**
 * 重置状态
 */
async function resetStatus() {
  try {
    await chrome.runtime.sendMessage({ type: 'RESET_STATUS' })

    currentState = {
      status: 'idle',
      currentPage: 0,
      totalPages: 0,
      asinCount: 0,
      error: null
    }

    updateUI()
    hideMessages()

    console.log('[Popup] 状态已重置')
  } catch (error) {
    console.error('[Popup] 重置失败:', error)
  }
}

/**
 * 更新UI
 */
function updateUI() {
  // 更新状态指示器
  elements.statusDot.className = `status-dot ${currentState.status}`

  // 更新状态文本
  const statusTextMap = {
    idle: '就绪',
    scraping: '正在采集...',
    paused: '已暂停',
    uploading: '正在上传...',
    completed: '完成',
    cancelled: '已取消',
    error: '错误'
  }
  elements.statusText.textContent = statusTextMap[currentState.status] || '未知'

  // 更新进度信息
  elements.currentPage.textContent = currentState.currentPage || '-'
  elements.totalPages.textContent =
    currentState.totalPages === '未知' ? '未知' : currentState.totalPages || '-'
  elements.asinCount.textContent = currentState.asinCount || 0

  // 更新进度条
  if (currentState.status === 'scraping' && currentState.totalPages > 0) {
    elements.progressBarContainer.style.display = 'block'
    const progress = (currentState.currentPage / currentState.totalPages) * 100
    elements.progressBar.style.width = `${progress}%`
  } else {
    elements.progressBarContainer.style.display = 'none'
  }

  // 更新按钮状态
  if (
    currentState.status === 'idle' ||
    currentState.status === 'error' ||
    currentState.status === 'completed' ||
    currentState.status === 'cancelled'
  ) {
    // 显示开始按钮
    elements.startButton.style.display = 'flex'
    elements.startButton.disabled = false
    elements.pauseButton.style.display = 'none'
    elements.resumeButton.style.display = 'none'
    elements.cancelButton.style.display = 'none'
  } else if (currentState.status === 'scraping') {
    // 显示暂停和取消按钮
    elements.startButton.style.display = 'none'
    elements.pauseButton.style.display = 'flex'
    elements.resumeButton.style.display = 'none'
    elements.cancelButton.style.display = 'flex'
  } else if (currentState.status === 'paused') {
    // 显示继续和取消按钮
    elements.startButton.style.display = 'none'
    elements.pauseButton.style.display = 'none'
    elements.resumeButton.style.display = 'flex'
    elements.cancelButton.style.display = 'flex'
  } else if (currentState.status === 'uploading') {
    // 上传中，禁用所有按钮
    elements.startButton.style.display = 'none'
    elements.pauseButton.style.display = 'none'
    elements.resumeButton.style.display = 'none'
    elements.cancelButton.style.display = 'none'
  }

  // 显示/隐藏重置按钮
  if (
    currentState.status === 'completed' ||
    currentState.status === 'error' ||
    currentState.status === 'cancelled'
  ) {
    elements.resetButton.style.display = 'flex'
  } else {
    elements.resetButton.style.display = 'none'
  }

  // 显示错误信息
  if (currentState.status === 'error' && currentState.error) {
    showErrorMessage(currentState.error)
  }
}

/**
 * 显示错误消息
 */
function showErrorMessage(message) {
  elements.errorMessage.textContent = message
  elements.errorMessage.style.display = 'block'
  elements.successMessage.style.display = 'none'
}

/**
 * 显示成功消息
 */
function showSuccessMessage(message) {
  elements.successMessage.textContent = message
  elements.successMessage.style.display = 'block'
  elements.errorMessage.style.display = 'none'
}

/**
 * 隐藏所有消息
 */
function hideMessages() {
  elements.errorMessage.style.display = 'none'
  elements.successMessage.style.display = 'none'
}

/**
 * 获取缓存的查询条件
 */
async function getCachedQueryParams() {
  try {
    const cached = await chrome.storage.local.get(['sellerStatsCache'])
    return cached.sellerStatsCache || null
  } catch (error) {
    console.error('[Popup] 获取缓存失败:', error)
    return null
  }
}

/**
 * 保存查询条件和结果到缓存
 */
async function saveCachedQueryParams(queryParams, result) {
  try {
    await chrome.storage.local.set({
      sellerStatsCache: {
        queryParams,
        result,
        timestamp: Date.now()
      }
    })
    console.log('[Popup] 查询结果已缓存')
  } catch (error) {
    console.error('[Popup] 保存缓存失败:', error)
  }
}

/**
 * 检查查询条件是否与缓存相同
 */
function isQueryParamsSame(params1, params2) {
  return (
    params1.minAsinCount === params2.minAsinCount &&
    params1.minOfferCount === params2.minOfferCount &&
    params1.apiEndpoint === params2.apiEndpoint
  )
}

/**
 * 恢复缓存的查询条件和结果
 */
async function restoreCachedQueryResult() {
  try {
    const cached = await getCachedQueryParams()
    if (!cached || !cached.queryParams || !cached.result) {
      return false
    }

    // 恢复查询条件到输入框
    const { minAsinCount, minOfferCount, apiEndpoint } = cached.queryParams

    // 只有当API端点匹配时才恢复（因为API端点可能已经在init中设置了）
    const currentEndpoint = elements.apiEndpoint.value.trim()
    if (apiEndpoint === currentEndpoint) {
      // 恢复查询条件
      elements.minAsinCount.value = minAsinCount || 1
      elements.minOfferCount.value = minOfferCount || 1

      // 显示缓存的结果
      displaySellerStats(cached.result)

      console.log('[Popup] 已恢复缓存的查询结果')
      return true
    }
  } catch (error) {
    console.error('[Popup] 恢复缓存失败:', error)
  }
  return false
}

/**
 * 检查当前查询条件是否与缓存匹配，如果不匹配则清除显示
 */
async function checkAndClearCacheIfNeeded() {
  try {
    const cached = await getCachedQueryParams()
    if (!cached || !cached.queryParams) {
      return
    }

    const currentQueryParams = {
      minAsinCount: parseInt(elements.minAsinCount.value) || 1,
      minOfferCount: parseInt(elements.minOfferCount.value) || 1,
      apiEndpoint: elements.apiEndpoint.value.trim()
    }

    // 如果条件不匹配，清除显示的结果
    if (!isQueryParamsSame(cached.queryParams, currentQueryParams)) {
      elements.statsResults.style.display = 'none'
      elements.statsError.style.display = 'none'
      elements.statsTableBody.innerHTML = ''
      elements.batchActions.style.display = 'none'
    }
  } catch (error) {
    console.error('[Popup] 检查缓存失败:', error)
  }
}

/**
 * 查询商家统计
 */
async function querySellerStatistics() {
  console.log('[Popup] 开始查询商家统计...')

  // 获取查询条件
  const minAsinCount = parseInt(elements.minAsinCount.value) || 1
  const minOfferCount = parseInt(elements.minOfferCount.value) || 1

  // 验证输入
  if (minAsinCount < 1) {
    alert('ASIN数量必须大于等于1')
    return
  }
  if (minOfferCount < 1) {
    alert('报价数量必须大于等于1')
    return
  }

  // 检查API配置
  const endpoint = elements.apiEndpoint.value.trim()
  if (!endpoint) {
    alert('请先配置API地址！')
    return
  }

  // 构建当前查询条件
  const currentQueryParams = {
    minAsinCount,
    minOfferCount,
    apiEndpoint: endpoint
  }

  // 检查缓存
  const cached = await getCachedQueryParams()
  if (cached && isQueryParamsSame(cached.queryParams, currentQueryParams)) {
    console.log('[Popup] 查询条件未改变，使用缓存结果')
    // 确保UI状态正确
    elements.statsLoading.style.display = 'none'
    elements.statsError.style.display = 'none'
    // 直接显示缓存的结果
    displaySellerStats(cached.result)
    return
  }

  try {
    // 显示加载状态
    elements.statsLoading.style.display = 'flex'
    elements.statsResults.style.display = 'none'
    elements.statsError.style.display = 'none'
    elements.querySellerStats.disabled = true

    // 构建消息
    const message = {
      type: 'QUERY_SELLER_STATS',
      data: {
        minAsinCount,
        minOfferCount,
        apiEndpoint: endpoint
      }
    }
    console.log('[Popup] 发送商家统计查询消息:', message)

    // 发送查询请求到background
    const response = await chrome.runtime.sendMessage(message)
    console.log('[Popup] 收到响应:', response)

    if (response.success) {
      // 保存查询条件和结果到缓存
      await saveCachedQueryParams(currentQueryParams, response.data)
      // 显示结果
      displaySellerStats(response.data)
    } else {
      throw new Error(response.error || '查询失败')
    }
  } catch (error) {
    console.error('[Popup] 查询商家统计失败:', error)
    elements.statsError.textContent = `查询失败: ${error.message}`
    elements.statsError.style.display = 'block'
    elements.statsResults.style.display = 'none'
  } finally {
    // 隐藏加载状态
    elements.statsLoading.style.display = 'none'
    elements.querySellerStats.disabled = false
  }
}

/**
 * 显示商家统计结果
 */
function displaySellerStats(data) {
  if (!data || !Array.isArray(data.sellers) || data.sellers.length === 0) {
    elements.statsError.textContent = '未找到符合条件的商家数据'
    elements.statsError.style.display = 'block'
    elements.statsResults.style.display = 'none'
    // 清空数据并禁用导出按钮
    elements.sellerStatsData = null
    if (elements.exportStatsData) {
      elements.exportStatsData.disabled = true
    }
    return
  }

  // 存储完整数据用于导出
  elements.sellerStatsData = data

  // 启用导出统计按钮
  if (elements.exportStatsData) {
    elements.exportStatsData.disabled = false
  }

  // 更新统计数量
  elements.statsCount.textContent = `${data.sellers.length} 条记录`

  // 清空表格
  elements.statsTableBody.innerHTML = ''

  // 填充数据
  data.sellers.forEach((seller, index) => {
    const row = document.createElement('tr')
    row.dataset.site = seller.site
    row.dataset.sellerId = seller.sellerId

    // 生成店铺标签HTML
    let shopsHtml = '<span class="text-muted">暂无店铺</span>'
    if (seller.shops && seller.shops.length > 0) {
      shopsHtml = seller.shops.map(shop => {
        const shopName = (shop.shop_name || '').replace(/"/g, '&quot;')
        const shopSite = shop.site || ''
        return `<span class="shop-tag" title="${shopName} (${shopSite})">${shopName} (${shopSite})</span>`
      }).join('')
    }

    row.innerHTML = `
      <td class="checkbox-cell">
        <input type="checkbox" class="row-checkbox" data-site="${seller.site}" data-seller-id="${seller.sellerId}">
      </td>
      <td class="site-tag-cell">
        <span class="site-badge">${seller.site || '-'}</span>
      </td>
      <td class="seller-id">${seller.sellerId || '-'}</td>
      <td class="asin-count">${seller.asinCount || 0}</td>
      <td class="offer-count">${seller.offerCount || 0}</td>
      <td class="shops-cell">
        <div class="shops-container">${shopsHtml}</div>
      </td>
      <td class="action-cell">
        <button class="btn-get-detail" data-site="${seller.site}" data-seller-id="${seller.sellerId}" title="获取商家详细信息">
          <span class="btn-icon">📋</span>
          获取详情
        </button>
      </td>
    `
    // 添加交替行样式
    if (index % 2 === 0) {
      row.style.backgroundColor = '#f8f9fa'
    }
    elements.statsTableBody.appendChild(row)
  })

  // 显示批量操作区域
  elements.batchActions.style.display = 'flex'

  // 绑定复选框事件
  const checkboxes = elements.statsTableBody.querySelectorAll('.row-checkbox')
  checkboxes.forEach(checkbox => {
    checkbox.addEventListener('change', updateBatchInfo)
  })

  // 绑定获取详情按钮事件
  const detailButtons = elements.statsTableBody.querySelectorAll('.btn-get-detail')
  detailButtons.forEach(btn => {
    btn.addEventListener('click', (e) => {
      const site = e.target.closest('.btn-get-detail').dataset.site
      const sellerId = e.target.closest('.btn-get-detail').dataset.sellerId
      getSellerDetail(site, sellerId)
    })
  })

  // 更新批量选择信息
  updateBatchInfo()

  // 检查是否有存储的数据，如果有则启用导出按钮
  checkStoredData()

  // 显示结果区域
  elements.statsResults.style.display = 'block'
  elements.statsError.style.display = 'none'
}

// 站点代码到域名的映射
const siteDomainMap = {
  US: 'amazon.com',
  UK: 'amazon.co.uk',
  DE: 'amazon.de',
  FR: 'amazon.fr',
  IT: 'amazon.it',
  ES: 'amazon.es',
  JP: 'amazon.co.jp',
  CA: 'amazon.ca',
  AU: 'amazon.com.au',
  MX: 'amazon.com.mx',
  BR: 'amazon.com.br',
  IN: 'amazon.in',
  SG: 'amazon.sg',
  AE: 'amazon.ae',
  SA: 'amazon.sa',
  NL: 'amazon.nl',
  PL: 'amazon.pl',
  SE: 'amazon.se',
  BE: 'amazon.com.be',
  IE: 'amazon.ie'
}

// 获取商家详情（带频率限制）
let lastFetchTime = 0
const FETCH_INTERVAL = 2000 // 2秒间隔，避免请求过快

/**
 * 全选/取消全选
 */
function handleSelectAll(e) {
  const checked = e.target.checked
  const checkboxes = elements.statsTableBody.querySelectorAll('.row-checkbox')
  checkboxes.forEach(checkbox => {
    checkbox.checked = checked
  })
  elements.selectAll.checked = checked
  elements.headerCheckbox.checked = checked
  updateBatchInfo()
}

/**
 * 更新批量选择信息
 */
function updateBatchInfo() {
  const checkboxes = elements.statsTableBody.querySelectorAll('.row-checkbox:checked')
  const count = checkboxes.length
  elements.batchInfo.textContent = `已选择 ${count} 项`
  elements.batchGetDetails.disabled = count === 0

  // 更新全选状态
  const allCheckboxes = elements.statsTableBody.querySelectorAll('.row-checkbox')
  const allChecked = allCheckboxes.length > 0 && checkboxes.length === allCheckboxes.length
  elements.selectAll.checked = allChecked
  elements.headerCheckbox.checked = allChecked

  // 检查存储的数据
  checkStoredData()
}

/**
 * 检查存储的数据，启用/禁用导出按钮
 */
async function checkStoredData() {
  try {
    const storedData = await chrome.storage.local.get(['sellerDetails'])
    const sellerDetails = storedData.sellerDetails || {}
    const hasData = Object.keys(sellerDetails).length > 0
    elements.exportSellerData.disabled = !hasData

    if (hasData) {
      const count = Object.keys(sellerDetails).length
      elements.exportSellerData.title = `导出 ${count} 条已存储的商家数据`
    } else {
      elements.exportSellerData.title = '没有可导出的数据'
    }
  } catch (error) {
    console.error('[Popup] 检查存储数据失败:', error)
  }
}

/**
 * 批量获取商家详情（优化版：并发处理）
 */
async function handleBatchGetDetails() {
  const checkboxes = elements.statsTableBody.querySelectorAll('.row-checkbox:checked')
  if (checkboxes.length === 0) {
    alert('请先选择要获取的商家')
    return
  }

  const sellers = Array.from(checkboxes).map(cb => ({
    site: cb.dataset.site,
    sellerId: cb.dataset.sellerId
  }))

  // 并发数量：根据总数动态调整，最多5个并发
  const concurrency = Math.min(5, Math.max(2, Math.ceil(sellers.length / 10)))

  if (!confirm(`确定要批量获取 ${sellers.length} 个商家的详细信息吗？\n\n将使用 ${concurrency} 个并发请求加速处理。`)) {
    return
  }

  // 显示进度
  elements.batchProgress.style.display = 'block'
  elements.batchGetDetails.disabled = true
  elements.exportSellerData.disabled = true

  let successCount = 0
  let failCount = 0
  let completedCount = 0
  const total = sellers.length
  const startTime = Date.now()

  // 进度更新防抖（避免频繁更新UI）
  let progressUpdateTimer = null
  const updateProgress = () => {
    if (progressUpdateTimer) {
      cancelAnimationFrame(progressUpdateTimer)
    }
    progressUpdateTimer = requestAnimationFrame(() => {
      const progress = (completedCount / total) * 100
      const elapsed = (Date.now() - startTime) / 1000 // 秒
      const speed = elapsed > 0 ? (completedCount / elapsed).toFixed(1) : '0.0'
      const remaining = completedCount > 0 && speed > 0 ? ((total - completedCount) / parseFloat(speed)).toFixed(0) : '-'

      elements.batchProgressText.textContent = `${completedCount} / ${total} (${speed}/秒${remaining !== '-' ? `, 剩余约${remaining}秒` : ''})`
      elements.batchProgressFill.style.width = `${progress}%`
    })
  }

  // 并发处理函数
  const processSeller = async (seller, index) => {
    try {
      // 添加小延迟避免同时发起过多请求（每个请求间隔100-300ms）
      if (index > 0) {
        await new Promise(resolve => setTimeout(resolve, 100 + (index % concurrency) * 50))
      }

      await getSellerDetailAndStore(seller.site, seller.sellerId)
      successCount++
      completedCount++
      updateProgress()
    } catch (error) {
      console.error(`[Popup] 获取商家 ${seller.sellerId} 失败:`, error)
      failCount++
      completedCount++
      updateProgress()
    }
  }

  // 分批并发处理
  const batches = []
  for (let i = 0; i < sellers.length; i += concurrency) {
    batches.push(sellers.slice(i, i + concurrency))
  }

  // 依次处理每个批次（批次间有短暂延迟）
  for (let batchIndex = 0; batchIndex < batches.length; batchIndex++) {
    const batch = batches[batchIndex]

    // 并发处理当前批次
    await Promise.all(batch.map((seller, index) => processSeller(seller, batchIndex * concurrency + index)))

    // 批次间延迟（避免请求过快）
    if (batchIndex < batches.length - 1) {
      await new Promise(resolve => setTimeout(resolve, 300))
    }
  }

  // 确保进度条更新到100%
  completedCount = total
  updateProgress()

  // 等待最后的UI更新完成
  await new Promise(resolve => setTimeout(resolve, 100))

  // 完成提示
  showSuccessMessage(`批量获取完成！成功: ${successCount} 个，失败: ${failCount} 个`)

  // 隐藏进度
  elements.batchProgress.style.display = 'none'
  elements.batchGetDetails.disabled = false
  updateBatchInfo()

  // 如果成功获取了数据，提示可以导出
  if (successCount > 0) {
    // 如果导出详情页面已打开，自动刷新显示最新数据
    if (elements.exportDetailModal && elements.exportDetailModal.style.display === 'flex') {
      refreshExportDetailPage()
    }

    setTimeout(() => {
      showSuccessMessage(`已成功获取 ${successCount} 个商家详情，可以点击"导出数据"按钮导出`)
    }, 1000)
  }
}

/**
 * 刷新导出详情页面数据（从存储中重新加载）
 */
async function refreshExportDetailPage() {
  try {
    // 获取存储的所有商家数据
    const storedData = await chrome.storage.local.get(['sellerDetails'])
    const sellerDetails = storedData.sellerDetails || {}

    // 更新统计信息
    const totalCount = Object.keys(sellerDetails).length
    if (elements.exportTotalCount) {
      elements.exportTotalCount.textContent = totalCount
    }
    if (elements.exportDetailCount) {
      elements.exportDetailCount.textContent = totalCount
    }

    // 获取最后更新时间
    let lastUpdate = null
    Object.values(sellerDetails).forEach(item => {
      if (item.fetchedAt) {
        const date = new Date(item.fetchedAt)
        if (!lastUpdate || date > lastUpdate) {
          lastUpdate = date
        }
      }
    })
    if (elements.exportLastUpdate) {
      elements.exportLastUpdate.textContent = lastUpdate
        ? new Date(lastUpdate).toLocaleString('zh-CN')
        : '-'
    }

    // 更新站点筛选器
    const sites = [...new Set(Object.keys(sellerDetails).map(key => key.split('|')[0]))].sort()
    if (elements.exportSiteFilter) {
      const currentSiteFilter = elements.exportSiteFilter.value
      elements.exportSiteFilter.innerHTML = '<option value="">所有站点</option>'
      sites.forEach(site => {
        const option = document.createElement('option')
        option.value = site
        option.textContent = site
        elements.exportSiteFilter.appendChild(option)
      })
      // 恢复之前的筛选
      elements.exportSiteFilter.value = currentSiteFilter
    }

    // 获取当前的搜索条件
    const currentSearch = elements.exportSearchInput ? elements.exportSearchInput.value.trim() : ''
    const currentSite = elements.exportSiteFilter ? elements.exportSiteFilter.value : ''

    // 渲染商家列表（保持当前的筛选条件）
    renderExportList(sellerDetails, { search: currentSearch, site: currentSite })
  } catch (error) {
    console.error('[Popup] 刷新导出页面失败:', error)
  }
}

/**
 * 显示导出详情页面
 */
async function showExportDetailPage() {
  try {
    // 获取存储的所有商家数据
    const storedData = await chrome.storage.local.get(['sellerDetails'])
    const sellerDetails = storedData.sellerDetails || {}

    if (Object.keys(sellerDetails).length === 0) {
      alert('没有可导出的数据，请先获取商家详情')
      return
    }

    // 显示模态框
    elements.exportDetailModal.style.display = 'flex'
    elements.exportDetailModal.classList.add('show')

    // 使用刷新函数更新数据（确保显示最新数据）
    await refreshExportDetailPage()
  } catch (error) {
    console.error('[Popup] 显示导出页面失败:', error)
    alert('显示导出页面失败: ' + error.message)
  }
}

/**
 * 渲染导出列表
 */
function renderExportList(sellerDetails, filter = {}) {
  const container = elements.exportListContainer
  container.innerHTML = ''

  const entries = Object.entries(sellerDetails)
    .filter(([key, value]) => {
      const [site, sellerId] = key.split('|')

      // 站点筛选
      if (filter.site && site !== filter.site) {
        return false
      }

      // 搜索筛选（包括店铺名称）
      if (filter.search) {
        const searchLower = filter.search.toLowerCase()
        const matchSellerId = sellerId.toLowerCase().includes(searchLower)
        const matchBusinessName = (value.businessName || '').toLowerCase().includes(searchLower)
        const matchEmail = (value.email || '').toLowerCase().includes(searchLower)
        const matchShop = value.shops && value.shops.some(shop =>
          (shop.shop_name || '').toLowerCase().includes(searchLower)
        )
        if (!matchSellerId && !matchBusinessName && !matchEmail && !matchShop) {
          return false
        }
      }

      return true
    })
    .sort((a, b) => {
      const [, dataA] = a
      const [, dataB] = b
      const dateA = dataA.fetchedAt ? new Date(dataA.fetchedAt) : new Date(0)
      const dateB = dataB.fetchedAt ? new Date(dataB.fetchedAt) : new Date(0)
      return dateB - dateA // 按时间倒序
    })

  if (entries.length === 0) {
    container.innerHTML = `
      <div class="export-empty">
        <div class="empty-icon">🔍</div>
        <p>没有找到匹配的商家数据</p>
      </div>
    `
    return
  }

  entries.forEach(([key, value]) => {
    const [site, sellerId] = key.split('|')
    const card = createSellerCard(site, sellerId, value)
    container.appendChild(card)
  })
}

/**
 * 创建商家卡片
 */
function createSellerCard(site, sellerId, data) {
  const card = document.createElement('div')
  card.className = 'seller-card'

  const hasData = data.businessName || data.email || data.phone

  card.innerHTML = `
    <div class="seller-card-header">
      <div class="seller-card-title">
        <span class="card-site-badge">${site}</span>
        <span class="card-seller-id">${sellerId}</span>
        ${hasData ? '<span class="card-status success">✓ 已获取</span>' : '<span class="card-status pending">⏳ 待获取</span>'}
      </div>
      <div class="seller-card-actions">
        ${data.url ? `<a href="${data.url}" target="_blank" class="card-link">查看页面 ↗</a>` : ''}
        ${data.fetchedAt ? `<span class="card-time">${new Date(data.fetchedAt).toLocaleString('zh-CN')}</span>` : ''}
      </div>
    </div>
    <div class="seller-card-body">
      ${data.businessName ? `
        <div class="card-field">
          <span class="field-label">Business Name:</span>
          <span class="field-value">${data.businessName}</span>
        </div>
      ` : ''}
      ${data.businessType ? `
        <div class="card-field">
          <span class="field-label">Business Type:</span>
          <span class="field-value">${data.businessType}</span>
        </div>
      ` : ''}
      ${data.tradeRegisterNumber ? `
        <div class="card-field">
          <span class="field-label">Trade Register Number:</span>
          <span class="field-value">${data.tradeRegisterNumber}</span>
        </div>
      ` : ''}
      ${data.phone ? `
        <div class="card-field">
          <span class="field-label">Phone:</span>
          <span class="field-value"><a href="tel:${data.phone}">${data.phone}</a></span>
        </div>
      ` : ''}
      ${data.email ? `
        <div class="card-field">
          <span class="field-label">Email:</span>
          <span class="field-value"><a href="mailto:${data.email}">${data.email}</a></span>
        </div>
      ` : ''}
      ${data.address && data.address.length > 0 ? `
        <div class="card-field">
          <span class="field-label">Address:</span>
          <div class="field-value address-field">
            ${data.address.map(line => `<div>${line}</div>`).join('')}
          </div>
        </div>
      ` : ''}
      ${data.shops && data.shops.length > 0 ? `
        <div class="card-field">
          <span class="field-label">关联店铺:</span>
          <div class="field-value shops-field">
            ${data.shops.map(shop => `
              <span class="shop-tag-small">
                ${(shop.shop_name || '').replace(/"/g, '&quot;')} (${shop.site || ''})
              </span>
            `).join('')}
          </div>
        </div>
      ` : ''}
    </div>
  `

  return card
}

/**
 * 筛选导出列表
 */
function filterExportList() {
  const search = elements.exportSearchInput.value.trim()
  const site = elements.exportSiteFilter.value

  const storedData = chrome.storage.local.get(['sellerDetails'], (result) => {
    const sellerDetails = result.sellerDetails || {}
    renderExportList(sellerDetails, { search, site })
  })
}

/**
 * 导出商家统计数据（包含店铺信息）
 */
async function exportStatsData() {
  try {
    if (!elements.sellerStatsData || !elements.sellerStatsData.sellers || elements.sellerStatsData.sellers.length === 0) {
      alert('没有可导出的统计数据，请先查询商家统计')
      return
    }

    // 选择导出格式
    const format = confirm('选择导出格式：\n确定 = JSON格式\n取消 = CSV格式') ? 'json' : 'csv'

    // 准备数据，包含店铺信息
    const dataArray = elements.sellerStatsData.sellers.map(seller => {
      // 格式化店铺信息
      let shopsText = ''
      let shopsJson = []
      if (seller.shops && seller.shops.length > 0) {
        shopsText = seller.shops.map(shop => `${shop.shop_name || ''} (${shop.site || ''})`).join(' | ')
        shopsJson = seller.shops
      }

      return {
        site: seller.site || '',
        sellerId: seller.sellerId || '',
        asinCount: seller.asinCount || 0,
        offerCount: seller.offerCount || 0,
        shops: shopsText, // CSV格式：文本
        shopsJson: shopsJson // JSON格式：数组
      }
    })

    if (format === 'json') {
      // 导出为JSON（包含完整的店铺数组）
      const jsonData = JSON.stringify({
        stats: elements.sellerStatsData.stats || {},
        sellers: dataArray.map(row => ({
          site: row.site,
          sellerId: row.sellerId,
          asinCount: row.asinCount,
          offerCount: row.offerCount,
          shops: row.shopsJson
        })),
        exportedAt: new Date().toISOString(),
        totalCount: dataArray.length
      }, null, 2)
      const blob = new Blob([jsonData], { type: 'application/json;charset=utf-8' })
      const url = URL.createObjectURL(blob)
      const a = document.createElement('a')
      a.href = url
      a.download = `seller-stats-${new Date().toISOString().split('T')[0]}.json`
      document.body.appendChild(a)
      a.click()
      document.body.removeChild(a)
      URL.revokeObjectURL(url)
      showSuccessMessage(`成功导出 ${dataArray.length} 条商家统计数据（JSON格式）`)
    } else {
      // 导出为CSV
      const headers = ['站点', '商家ID', 'ASIN数量', '报价数量', '关联店铺']
      const csvRows = [
        headers.join(','),
        ...dataArray.map(row => [
          row.site || '',
          row.sellerId || '',
          row.asinCount || 0,
          row.offerCount || 0,
          `"${(row.shops || '').replace(/"/g, '""')}"`
        ].join(','))
      ]
      const csvData = csvRows.join('\n')
      const blob = new Blob(['\ufeff' + csvData], { type: 'text/csv;charset=utf-8;' }) // BOM for Excel
      const url = URL.createObjectURL(blob)
      const a = document.createElement('a')
      a.href = url
      a.download = `seller-stats-${new Date().toISOString().split('T')[0]}.csv`
      document.body.appendChild(a)
      a.click()
      document.body.removeChild(a)
      URL.revokeObjectURL(url)
      showSuccessMessage(`成功导出 ${dataArray.length} 条商家统计数据（CSV格式）`)
    }
  } catch (error) {
    console.error('[Popup] 导出统计数据失败:', error)
    showErrorMessage(`导出失败: ${error.message}`)
  }
}

/**
 * 导出商家数据
 */
async function handleExportData(format = null) {
  try {
    // 获取存储的所有商家数据
    const storedData = await chrome.storage.local.get(['sellerDetails'])
    const sellerDetails = storedData.sellerDetails || {}

    if (Object.keys(sellerDetails).length === 0) {
      alert('没有可导出的数据，请先获取商家详情')
      return
    }

    // 如果没有指定格式，让用户选择
    if (!format) {
      format = confirm('选择导出格式：\n确定 = JSON格式\n取消 = CSV格式') ? 'json' : 'csv'
    }

    // 转换为数组格式
    const dataArray = Object.entries(sellerDetails).map(([key, value]) => {
      const [site, sellerId] = key.split('|')

      // 格式化店铺信息
      let shopsText = ''
      let shopsJson = []
      if (value.shops && value.shops.length > 0) {
        shopsText = value.shops.map(shop => `${shop.shop_name || ''} (${shop.site || ''})`).join(' | ')
        shopsJson = value.shops
      }

      return {
        site,
        sellerId,
        businessName: value.businessName || '',
        businessType: value.businessType || '',
        tradeRegisterNumber: value.tradeRegisterNumber || '',
        phone: value.phone || '',
        email: value.email || '',
        address: value.address ? value.address.join(' | ') : '',
        addressLines: value.address || [],
        shops: shopsText, // CSV格式：文本
        shopsJson: shopsJson, // JSON格式：数组
        url: value.url || '',
        fetchedAt: value.fetchedAt || ''
      }
    })

    if (format === 'json') {
      // 导出为JSON（包含完整的店铺数组）
      const jsonData = JSON.stringify(
        dataArray.map(row => ({
          site: row.site,
          sellerId: row.sellerId,
          businessName: row.businessName,
          businessType: row.businessType,
          tradeRegisterNumber: row.tradeRegisterNumber,
          phone: row.phone,
          email: row.email,
          address: row.addressLines,
          shops: row.shopsJson, // 完整的店铺数组
          url: row.url,
          fetchedAt: row.fetchedAt
        })),
        null,
        2
      )
      const blob = new Blob([jsonData], { type: 'application/json;charset=utf-8' })
      const url = URL.createObjectURL(blob)
      const a = document.createElement('a')
      a.href = url
      a.download = `seller-details-${new Date().toISOString().split('T')[0]}.json`
      document.body.appendChild(a)
      a.click()
      document.body.removeChild(a)
      URL.revokeObjectURL(url)
    } else {
      // 导出为CSV（包含店铺信息）
      const headers = ['站点', '商家ID', 'Business Name', 'Business Type', 'Trade Register Number', 'Phone', 'Email', 'Address', '关联店铺', 'URL', '获取时间']
      const csvRows = [
        headers.join(','),
        ...dataArray.map(row => [
          row.site || '',
          row.sellerId || '',
          `"${(row.businessName || '').replace(/"/g, '""')}"`,
          `"${(row.businessType || '').replace(/"/g, '""')}"`,
          `"${(row.tradeRegisterNumber || '').replace(/"/g, '""')}"`,
          `"${(row.phone || '').replace(/"/g, '""')}"`,
          `"${(row.email || '').replace(/"/g, '""')}"`,
          `"${(row.address || '').replace(/"/g, '""')}"`,
          `"${(row.shops || '').replace(/"/g, '""')}"`,
          `"${(row.url || '').replace(/"/g, '""')}"`,
          `"${(row.fetchedAt || '').replace(/"/g, '""')}"`
        ].join(','))
      ]
      const csvData = csvRows.join('\n')
      const blob = new Blob(['\ufeff' + csvData], { type: 'text/csv;charset=utf-8;' }) // BOM for Excel
      const url = URL.createObjectURL(blob)
      const a = document.createElement('a')
      a.href = url
      a.download = `seller-details-${new Date().toISOString().split('T')[0]}.csv`
      document.body.appendChild(a)
      a.click()
      document.body.removeChild(a)
      URL.revokeObjectURL(url)
    }

    showSuccessMessage(`成功导出 ${dataArray.length} 条商家数据（${format.toUpperCase()}格式）`)
  } catch (error) {
    console.error('[Popup] 导出数据失败:', error)
    alert('导出数据失败: ' + error.message)
  }
}

/**
 * 获取商家详情并存储（不打开新页面）
 */
async function getSellerDetailAndStore(site, sellerId) {
  if (!site || !sellerId) {
    throw new Error('站点或商家ID缺失')
  }

  // 获取站点域名
  const domain = siteDomainMap[site]
  if (!domain) {
    throw new Error(`未找到站点 ${site} 对应的域名`)
  }

  // 构建卖家详情页面URL
  const sellerUrl = `https://www.${domain}/sp?ie=UTF8&seller=${sellerId}`

  try {
    // 通过background script获取页面内容
    const response = await chrome.runtime.sendMessage({
      type: 'FETCH_SELLER_DETAIL',
      data: { url: sellerUrl, site, sellerId }
    })

    if (response.success && response.data) {
      // 从统计数据中查找店铺信息
      let shops = []
      if (elements.sellerStatsData && elements.sellerStatsData.sellers) {
        const sellerInfo = elements.sellerStatsData.sellers.find(
          s => s.site === site && s.sellerId === sellerId
        )
        if (sellerInfo && sellerInfo.shops) {
          shops = sellerInfo.shops
        }
      }

      // 存储到浏览器
      const storageKey = `sellerDetails`
      const storedData = await chrome.storage.local.get([storageKey])
      const sellerDetails = storedData[storageKey] || {}

      sellerDetails[`${site}|${sellerId}`] = {
        ...response.data,
        shops: shops, // 保存店铺信息
        fetchedAt: new Date().toISOString(),
        url: sellerUrl
      }

      await chrome.storage.local.set({ [storageKey]: sellerDetails })
      console.log(`[Popup] 商家 ${sellerId} 详情已存储（包含 ${shops.length} 个店铺）`)

      // 更新导出按钮状态
      checkStoredData()

      // 注意：单个获取时不立即刷新，避免频繁刷新
      // 批量获取完成后会统一刷新

      return response.data
    } else {
      throw new Error(response.error || '无法获取商家信息')
    }
  } catch (error) {
    console.error('[Popup] 获取商家详情失败:', error)
    throw error
  }
}

/**
 * 获取商家详细信息（单个，显示在模态框）
 */
async function getSellerDetail(site, sellerId) {
  if (!site || !sellerId) {
    alert('站点或商家ID缺失')
    return
  }

  // 频率限制检查
  const now = Date.now()
  if (now - lastFetchTime < FETCH_INTERVAL) {
    const waitTime = Math.ceil((FETCH_INTERVAL - (now - lastFetchTime)) / 1000)
    alert(`请稍候 ${waitTime} 秒后再试，避免请求过快`)
    return
  }
  lastFetchTime = now

  try {
    // 显示模态框和加载状态
    elements.sellerDetailModal.style.display = 'flex'
    elements.sellerDetailModal.classList.add('show')
    elements.sellerDetailContent.innerHTML = `
      <div class="seller-detail-loading">
        <div class="loading-spinner"></div>
        <span>正在获取商家信息...</span>
      </div>
    `

    // 获取并存储商家信息
    const data = await getSellerDetailAndStore(site, sellerId)

    // 如果导出详情页面已打开，刷新显示最新数据
    if (elements.exportDetailModal && elements.exportDetailModal.style.display === 'flex') {
      refreshExportDetailPage()
    }

    // 显示商家详情
    displaySellerDetail(data, site, sellerId)
  } catch (error) {
    console.error('[Popup] 获取商家信息失败:', error)
    const domain = siteDomainMap[site]
    const sellerUrl = `https://www.${domain}/sp?ie=UTF8&seller=${sellerId}`

    elements.sellerDetailContent.innerHTML = `
      <div class="seller-detail-error">
        <p>❌ 获取商家信息失败</p>
        <p>${error.message}</p>
        <div class="seller-detail-url">
          <a href="${sellerUrl}" target="_blank">${sellerUrl}</a>
        </div>
      </div>
    `
  }
}

/**
 * 显示商家详细信息
 */
function displaySellerDetail(data, site, sellerId) {
  const domain = siteDomainMap[site]
  const sellerUrl = `https://www.${domain}/sp?ie=UTF8&seller=${sellerId}`

  let html = `
    <div class="seller-detail-info">
      <div class="detail-header">
        <div class="detail-site-badge">${site}</div>
        <div class="detail-seller-id">${sellerId}</div>
        <a href="${sellerUrl}" target="_blank" class="detail-link">查看原页面 ↗</a>
      </div>
  `

  if (data.businessName) {
    html += `
      <div class="detail-item">
        <div class="detail-label">Business Name:</div>
        <div class="detail-value">${data.businessName}</div>
      </div>
    `
  }

  if (data.businessType) {
    html += `
      <div class="detail-item">
        <div class="detail-label">Business Type:</div>
        <div class="detail-value">${data.businessType}</div>
      </div>
    `
  }

  if (data.tradeRegisterNumber) {
    html += `
      <div class="detail-item">
        <div class="detail-label">Trade Register Number:</div>
        <div class="detail-value">${data.tradeRegisterNumber}</div>
      </div>
    `
  }

  if (data.phone) {
    html += `
      <div class="detail-item">
        <div class="detail-label">Phone:</div>
        <div class="detail-value">${data.phone}</div>
      </div>
    `
  }

  if (data.email) {
    html += `
      <div class="detail-item">
        <div class="detail-label">Email:</div>
        <div class="detail-value"><a href="mailto:${data.email}">${data.email}</a></div>
      </div>
    `
  }

  if (data.address && data.address.length > 0) {
    html += `
      <div class="detail-item">
        <div class="detail-label">Business Address:</div>
        <div class="detail-value address-lines">
          ${data.address.map(line => `<div>${line}</div>`).join('')}
        </div>
      </div>
    `
  }

  html += `
      </div>
  `

  elements.sellerDetailContent.innerHTML = html
}

// 初始化popup
document.addEventListener('DOMContentLoaded', () => {
  // 设置版本号
  const manifest = chrome.runtime.getManifest()
  const versionBadge = document.getElementById('versionBadge')
  if (versionBadge && manifest.version) {
    versionBadge.textContent = `v${manifest.version}`
  }

  // 调用init函数
  init()
})

console.log('[Popup] 脚本已加载')
