/**
 * 内容脚本
 * 在亚马逊页面中运行，负责提取ASIN和处理分页
 */

// 亚马逊站点配置
const AMAZON_SITES = [
  { id: 203, code: 'IE', name: '爱尔兰', currency: 'EUR', domain: 'amazon.ie' },
  { id: 1, code: 'US', name: '美国', currency: 'USD', domain: 'amazon.com' },
  { id: 2, code: 'UK', name: '英国', currency: 'GBP', domain: 'amazon.co.uk' },
  { id: 3, code: 'DE', name: '德国', currency: 'EUR', domain: 'amazon.de' },
  { id: 4, code: 'FR', name: '法国', currency: 'EUR', domain: 'amazon.fr' },
  { id: 5, code: 'IT', name: '意大利', currency: 'EUR', domain: 'amazon.it' },
  { id: 6, code: 'ES', name: '西班牙', currency: 'EUR', domain: 'amazon.es' },
  { id: 7, code: 'JP', name: '日本', currency: 'JPY', domain: 'amazon.co.jp' },
  { id: 8, code: 'CA', name: '加拿大', currency: 'CAD', domain: 'amazon.ca' },
  { id: 9, code: 'AU', name: '澳大利亚', currency: 'AUD', domain: 'amazon.com.au' },
  { id: 10, code: 'MX', name: '墨西哥', currency: 'MXN', domain: 'amazon.com.mx' },
  { id: 11, code: 'BR', name: '巴西', currency: 'BRL', domain: 'amazon.com.br' },
  { id: 12, code: 'IN', name: '印度', currency: 'INR', domain: 'amazon.in' },
  { id: 13, code: 'SG', name: '新加坡', currency: 'SGD', domain: 'amazon.sg' },
  { id: 14, code: 'AE', name: '阿联酋', currency: 'AED', domain: 'amazon.ae' },
  { id: 15, code: 'SA', name: '沙特阿拉伯', currency: 'SAR', domain: 'amazon.sa' },
  { id: 121, code: 'NL', name: '荷兰', currency: 'EUR', domain: 'amazon.nl' },
  { id: 122, code: 'PL', name: '波兰', currency: 'PLN', domain: 'amazon.pl' },
  { id: 123, code: 'SE', name: '瑞典', currency: 'SEK', domain: 'amazon.se' },
  { id: 124, code: 'BE', name: '比利时', currency: 'EUR', domain: 'amazon.com.be' }
]

/**
 * 根据主机名获取站点信息
 */
function getSiteByHostname(hostname) {
  if (!hostname) return null

  const normalizedHost = hostname.toLowerCase()

  for (const site of AMAZON_SITES) {
    if (normalizedHost.includes(site.domain)) {
      return site
    }
  }

  return null
}

// 动态导入模块（使用相对路径）
const AsinExtractor = (() => {
  // 简化的ASIN提取器 - 直接从亚马逊原生DOM提取
  class AsinExtractor {
    static async extractFromPage() {
      const asins = new Set()

      console.log('[ASIN Extractor] ========== 开始提取ASIN ==========')

      // 优先在 sg-col-inner 区域内查找（亚马逊商品列表区域）
      const searchArea = document.querySelector('.sg-col-inner')

      if (searchArea) {
        console.log('[ASIN Extractor] ✓ 找到商品列表区域 (.sg-col-inner)')

        // 在此区域内查找data-asin
        const elementsWithDataAsin = searchArea.querySelectorAll('[data-asin]')
        console.log(`[ASIN Extractor] 找到 ${elementsWithDataAsin.length} 个带 data-asin 的元素`)

        // 详细日志：显示前几个元素的信息
        if (elementsWithDataAsin.length > 0) {
          console.log(`[ASIN Extractor] 前3个元素示例:`)
          Array.from(elementsWithDataAsin)
            .slice(0, 3)
            .forEach((el, i) => {
              const asin = el.getAttribute('data-asin')
              console.log(
                `  ${i + 1}. ASIN: ${asin}, 标签: ${el.tagName}, role: ${el.getAttribute('role')}`
              )
            })
        }

        elementsWithDataAsin.forEach(el => {
          const asin = el.getAttribute('data-asin')
          if (asin && this.isValidAsin(asin)) {
            asins.add(asin)
          }
        })
      } else {
        console.warn('[ASIN Extractor] ⚠️ 警告：未找到 .sg-col-inner 区域，使用全页面搜索')

        // 备用方案：在整个页面查找
        const elementsWithDataAsin = document.querySelectorAll('[data-asin]')
        console.log(`[ASIN Extractor] 找到 ${elementsWithDataAsin.length} 个带 data-asin 的元素`)

        elementsWithDataAsin.forEach(el => {
          const asin = el.getAttribute('data-asin')
          if (asin && this.isValidAsin(asin)) {
            asins.add(asin)
          }
        })
      }

      // 额外检查：从role=listitem元素提取
      const listItems = document.querySelectorAll('[role="listitem"][data-asin]')
      console.log(`[ASIN Extractor] 找到 ${listItems.length} 个 listitem 元素`)

      listItems.forEach(item => {
        const asin = item.getAttribute('data-asin')
        if (asin && this.isValidAsin(asin)) {
          asins.add(asin)
        }
      })

      const asinArray = Array.from(asins)

      console.log(`[ASIN Extractor] ========== 提取完成 ==========`)
      console.log(`[ASIN Extractor] ✓ 提取到 ${asinArray.length} 个唯一ASIN`)

      if (asinArray.length > 0) {
        console.log(`[ASIN Extractor] 示例ASIN:`, asinArray.slice(0, 3))
      } else {
        console.error('[ASIN Extractor] ❌ 错误：未能提取到任何ASIN！')
        console.log('[ASIN Extractor] 请检查：')
        console.log('  1. 当前在亚马逊商品列表页')
        console.log('  2. 页面已完全加载')
        console.log('  3. 能看到商品列表')
        console.log('  4. 滚动页面触发懒加载')
      }

      return asinArray
    }

    static extractAsinFromUrl(url) {
      if (!url) return null

      const patterns = [
        /\/dp\/([A-Z0-9]{10})/,
        /\/gp\/product\/([A-Z0-9]{10})/,
        /ASIN=([A-Z0-9]{10})/,
        /asin=([A-Z0-9]{10})/i
      ]

      for (const pattern of patterns) {
        const match = url.match(pattern)
        if (match && match[1]) {
          return match[1]
        }
      }

      return null
    }

    static isValidAsin(asin) {
      if (!asin || typeof asin !== 'string') return false

      asin = asin.trim()

      // ASIN是10位字符，由大写字母和数字组成
      const asinRegex = /^[A-Z0-9]{10}$/

      if (!asinRegex.test(asin)) {
        return false
      }

      // 排除全数字
      if (/^\d{10}$/.test(asin)) {
        return false
      }

      return true
    }
  }
  return AsinExtractor
})()

const PaginationHandler = (() => {
  class PaginationHandler {
    static hasNextPage() {
      const nextButton = this.getNextPageButton()
      if (nextButton) {
        // 检查按钮是否被禁用
        const isDisabled =
          nextButton.classList.contains('s-pagination-disabled') ||
          nextButton.classList.contains('disabled') ||
          nextButton.hasAttribute('disabled') ||
          nextButton.getAttribute('aria-disabled') === 'true'

        console.log(`[Pagination] Next按钮状态: ${isDisabled ? '禁用' : '可用'}`)
        return !isDisabled
      }

      return false
    }

    static getNextPageButton() {
      // 亚马逊底部分页选择器
      const selectors = [
        'a.s-pagination-next',
        '.s-pagination-item.s-pagination-next',
        'a[aria-label="Next"]',
        'a[aria-label="Go to next page"]'
      ]

      for (const selector of selectors) {
        const button = document.querySelector(selector)
        if (button) {
          console.log(`[Pagination] 找到下一页按钮: ${selector}`)
          return button
        }
      }

      console.warn('[Pagination] 未找到下一页按钮')
      return null
    }

    static async goToNextPage() {
      const nextButton = this.getNextPageButton()
      if (!nextButton) {
        console.log('[Pagination] 没有找到下一页按钮')
        return false
      }

      if (!this.hasNextPage()) {
        console.log('[Pagination] 已经是最后一页')
        return false
      }

      try {
        const currentUrl = window.location.href
        const currentPageNumber = this.getCurrentPageNumber()

        console.log(`[Pagination] 点击下一页按钮，当前页码: ${currentPageNumber}`)

        nextButton.click()

        await this.waitForPageLoad(currentUrl, currentPageNumber)

        console.log('[Pagination] 成功进入下一页')
        return true
      } catch (error) {
        console.error('[Pagination] 进入下一页失败:', error)
        return false
      }
    }

    static getCurrentPageNumber() {
      // 方法1: 从亚马逊分页组件获取当前页
      const activePageElement = document.querySelector('.s-pagination-item.s-pagination-selected')
      if (activePageElement) {
        const pageText = activePageElement.textContent.trim()
        const pageNum = parseInt(pageText, 10)
        if (!isNaN(pageNum)) {
          console.log(`[Pagination] 当前页码: ${pageNum}`)
          return pageNum
        }
      }

      // 方法2: 从URL参数获取
      const urlParams = new URLSearchParams(window.location.search)
      const pageFromUrl = urlParams.get('page') || urlParams.get('p')
      if (pageFromUrl) {
        const pageNum = parseInt(pageFromUrl, 10)
        if (!isNaN(pageNum)) {
          console.log(`[Pagination] 从URL获取页码: ${pageNum}`)
          return pageNum
        }
      }

      // 方法3: 查找所有页码按钮中的选中状态
      const allPageButtons = document.querySelectorAll('.s-pagination-item')
      for (const btn of allPageButtons) {
        if (
          btn.classList.contains('s-pagination-selected') ||
          btn.classList.contains('s-pagination-disabled')
        ) {
          const pageText = btn.textContent.trim()
          const pageNum = parseInt(pageText, 10)
          if (!isNaN(pageNum)) {
            return pageNum
          }
        }
      }

      return 1
    }

    static async waitForPageLoad(previousUrl, previousPage) {
      const maxWaitTime = 10000
      const checkInterval = 200
      let elapsedTime = 0

      return new Promise((resolve, reject) => {
        const checkPageChanged = setInterval(() => {
          elapsedTime += checkInterval

          const currentUrl = window.location.href
          const currentPage = this.getCurrentPageNumber()

          if (currentUrl !== previousUrl || currentPage > previousPage) {
            clearInterval(checkPageChanged)
            setTimeout(() => resolve(), 2000)
            return
          }

          if (elapsedTime >= maxWaitTime) {
            clearInterval(checkPageChanged)
            reject(new Error('页面加载超时'))
          }
        }, checkInterval)
      })
    }

    static getTotalPages() {
      // 从亚马逊底部分页获取最大页码
      const pageButtons = document.querySelectorAll('.s-pagination-item')
      let maxPage = 0

      console.log(`[Pagination] 找到 ${pageButtons.length} 个分页元素`)

      pageButtons.forEach(btn => {
        // 跳过Previous、Next和省略号
        if (
          btn.classList.contains('s-pagination-previous') ||
          btn.classList.contains('s-pagination-next') ||
          btn.classList.contains('s-pagination-ellipsis')
        ) {
          return
        }

        const pageText = btn.textContent.trim()
        const pageNum = parseInt(pageText, 10)

        if (!isNaN(pageNum) && pageNum > maxPage) {
          maxPage = pageNum
          console.log(`[Pagination] 发现页码: ${pageNum}`)
        }
      })

      if (maxPage > 0) {
        console.log(`[Pagination] ✓ 总页数: ${maxPage}`)
        return maxPage
      }

      // 尝试从结果统计估算
      const resultStats = document.querySelector('.s-desktop-toolbar, .s-result-count')
      if (resultStats) {
        const text = resultStats.textContent
        const match = text.match(/of\s+([\d,]+)\s+results/i)
        if (match) {
          const totalResults = parseInt(match[1].replace(/,/g, ''), 10)
          const itemsPerPage = 16
          const estimatedPages = Math.ceil(totalResults / itemsPerPage)
          console.log(`[Pagination] 估算总页数: ${estimatedPages}`)
          return estimatedPages
        }
      }

      console.log('[Pagination] ⚠️ 无法确定总页数')
      return null
    }
  }
  return PaginationHandler
})()

// 爬取状态
let isScrapingActive = false
let isPaused = false
let isCancelled = false
let collectedAsins = []

/**
 * 执行ASIN采集任务
 */
async function startScraping() {
  if (isScrapingActive) {
    console.log('[Content Script] 采集任务已在进行中')
    return
  }

  isScrapingActive = true
  isPaused = false
  isCancelled = false
  collectedAsins = []

  try {
    console.log('[Content Script] ==================== 开始采集 ====================')
    console.log('[Content Script] 页面URL:', window.location.href)

    let currentPage = 1
    const totalPages = PaginationHandler.getTotalPages() || '未知'

    console.log(`[Content Script] 当前页: ${currentPage}, 总页数: ${totalPages}`)

    // 发送开始消息
    chrome.runtime.sendMessage(
      {
        type: 'SCRAPING_PROGRESS',
        data: {
          currentPage,
          totalPages,
          asinCount: 0,
          status: 'scraping'
        }
      },
      () => {
        if (chrome.runtime.lastError) {
          // 忽略错误
        }
      }
    )

    // 采集第一页
    console.log(`[Content Script] 开始采集第 ${currentPage} 页...`)
    const firstPageAsins = await AsinExtractor.extractFromPage()

    if (firstPageAsins.length === 0) {
      console.error('[Content Script] ❌ 错误：第一页未采集到任何ASIN！')
      throw new Error(
        '页面上没有找到ASIN数据，请确认：\n1. 当前在亚马逊商品列表页\n2. 页面已完全加载\n3. 可以看到商品列表'
      )
    }

    collectedAsins.push(...firstPageAsins)

    console.log(
      `[Content Script] ✓ 第${currentPage}页采集完成，获取 ${firstPageAsins.length} 个ASIN`
    )

    // 发送进度更新
    chrome.runtime.sendMessage(
      {
        type: 'SCRAPING_PROGRESS',
        data: {
          currentPage,
          totalPages,
          asinCount: collectedAsins.length,
          status: 'scraping'
        }
      },
      () => {
        if (chrome.runtime.lastError) {
          // 忽略错误
        }
      }
    )

    // 遍历所有页面
    while (PaginationHandler.hasNextPage() && !isCancelled) {
      // 检查暂停状态
      while (isPaused && !isCancelled) {
        console.log('[Content Script] 采集已暂停')
        await new Promise(resolve => setTimeout(resolve, 1000))
      }

      if (isCancelled) {
        console.log('[Content Script] 采集已取消')
        break
      }

      currentPage++

      console.log(`[Content Script] 准备采集第${currentPage}页`)

      const success = await PaginationHandler.goToNextPage()

      if (!success) {
        console.warn('[Content Script] 无法进入下一页，停止采集')
        break
      }

      // 采集当前页
      const pageAsins = await AsinExtractor.extractFromPage()
      collectedAsins.push(...pageAsins)

      console.log(`[Content Script] 第${currentPage}页采集完成，获取 ${pageAsins.length} 个ASIN`)

      // 发送进度更新
      chrome.runtime.sendMessage(
        {
          type: 'SCRAPING_PROGRESS',
          data: {
            currentPage,
            totalPages,
            asinCount: collectedAsins.length,
            status: isPaused ? 'paused' : 'scraping'
          }
        },
        () => {
          if (chrome.runtime.lastError) {
            // 忽略错误
          }
        }
      )
    }

    // 去重
    const uniqueAsins = Array.from(new Set(collectedAsins))

    // 获取完整的页面信息
    const pageInfo = getPageInfo()

    if (isCancelled) {
      console.log(`[Content Script] 采集已取消，已采集 ${uniqueAsins.length} 个ASIN`)
      chrome.runtime.sendMessage(
        {
          type: 'SCRAPING_CANCELLED',
          data: {
            asins: uniqueAsins,
            totalCount: uniqueAsins.length,
            totalPages: currentPage,
            pageInfo: pageInfo // 包含完整页面信息
          }
        },
        () => {
          if (chrome.runtime.lastError) {
            // 忽略错误
          }
        }
      )
    } else {
      console.log(`[Content Script] 采集完成，共采集 ${uniqueAsins.length} 个唯一ASIN`)
      chrome.runtime.sendMessage(
        {
          type: 'SCRAPING_COMPLETE',
          data: {
            asins: uniqueAsins,
            totalCount: uniqueAsins.length,
            totalPages: currentPage,
            pageInfo: pageInfo // 包含完整页面信息
          }
        },
        () => {
          if (chrome.runtime.lastError) {
            // 忽略错误
          }
        }
      )
    }
  } catch (error) {
    console.error('[Content Script] 采集过程中发生错误:', error)

    // 发送错误消息
    chrome.runtime.sendMessage(
      {
        type: 'SCRAPING_ERROR',
        data: {
          error: error.message
        }
      },
      () => {
        if (chrome.runtime.lastError) {
          // 忽略错误
        }
      }
    )
  } finally {
    isScrapingActive = false
    isPaused = false
    isCancelled = false
  }
}

/**
 * 获取页面完整信息
 */
function getPageInfo() {
  const info = {
    // 基本信息
    site: null, // 站点（com, de, co.uk等）
    marketplace: null, // 市场ID
    url: window.location.href,

    // 分类和搜索
    category: null,
    categoryId: null,
    searchKeyword: null,

    // 统计信息
    resultCount: null,
    totalResults: null,

    // 卖家信息
    sellerId: null,
    sellerName: null,

    // 时间戳
    timestamp: new Date().toISOString()
  }

  try {
    // 解析站点信息
    const hostname = window.location.hostname
    const siteInfo = getSiteByHostname(hostname)

    if (siteInfo) {
      info.site = siteInfo.code
      info.siteName = siteInfo.name
      info.siteId = siteInfo.id
      info.currency = siteInfo.currency
    } else {
      // 降级处理：从hostname提取
      info.site = hostname.replace('www.amazon.', '').replace('.com', '').toUpperCase()
    }

    // 从URL获取marketplace ID
    const urlParams = new URLSearchParams(window.location.search)
    info.marketplace = urlParams.get('marketplaceID') || null

    // 获取分类信息
    const categoryElement =
      document.querySelector('#searchDropdownBox option[selected]') ||
      document.querySelector('.a-breadcrumb li:last-child')

    if (categoryElement) {
      info.category = categoryElement.textContent.trim()
    }

    // 从URL获取分类ID
    const rhParam = urlParams.get('rh')
    if (rhParam) {
      const nodeMatch = rhParam.match(/n:(\d+)/)
      if (nodeMatch) {
        info.categoryId = nodeMatch[1]
        if (!info.category) {
          info.category = `Category ${nodeMatch[1]}`
        }
      }
    }

    // 获取搜索关键词
    const searchBox = document.querySelector('#twotabsearchtextbox, input[name="field-keywords"]')
    if (searchBox && searchBox.value) {
      info.searchKeyword = searchBox.value.trim()
    }

    // 从URL获取搜索词
    if (!info.searchKeyword) {
      info.searchKeyword = urlParams.get('k') || urlParams.get('keywords') || null
    }

    // 获取卖家信息
    info.sellerId = urlParams.get('me') || null

    // 尝试从页面获取卖家名称
    const sellerLink = document.querySelector('a[href*="/sp?seller="]')
    if (sellerLink) {
      info.sellerName = sellerLink.textContent.trim()
    }

    // 获取结果数量
    const resultText = document.querySelector(
      '.s-desktop-toolbar .sg-col-inner h1, .s-result-count, .a-size-base'
    )
    if (resultText) {
      const text = resultText.textContent

      // 匹配 "1-16 of 498 results"
      const matchResults = text.match(/(\d+)-(\d+)\s+of\s+([\d,]+)\s+results/i)
      if (matchResults) {
        info.resultCount = matchResults[2] // 当前显示到第几个
        info.totalResults = matchResults[3].replace(/,/g, '') // 总结果数
      } else {
        const simpleMatch = text.match(/([\d,]+)/)
        if (simpleMatch) {
          info.resultCount = simpleMatch[1].replace(/,/g, '')
        }
      }
    }

    console.log('[Content Script] 页面信息:', info)
  } catch (error) {
    console.error('[Content Script] 获取页面信息失败:', error)
  }

  return info
}

// 监听来自popup或background的消息
chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
  console.log('[Content Script] 收到消息:', message)

  switch (message.type) {
    case 'GET_PAGE_INFO':
      sendResponse({ success: true, data: getPageInfo() })
      break

    case 'START_SCRAPING':
      startScraping()
      sendResponse({ success: true })
      break

    case 'PAUSE_SCRAPING':
      if (isScrapingActive) {
        isPaused = true
        console.log('[Content Script] 暂停采集')
        sendResponse({ success: true, paused: true })
      } else {
        sendResponse({ success: false, error: '没有正在进行的采集任务' })
      }
      break

    case 'RESUME_SCRAPING':
      if (isScrapingActive && isPaused) {
        isPaused = false
        console.log('[Content Script] 恢复采集')
        sendResponse({ success: true, resumed: true })
      } else {
        sendResponse({ success: false, error: '没有暂停的采集任务' })
      }
      break

    case 'CANCEL_SCRAPING':
      if (isScrapingActive) {
        isCancelled = true
        isPaused = false
        console.log('[Content Script] 取消采集')
        sendResponse({ success: true, cancelled: true })
      } else {
        sendResponse({ success: false, error: '没有正在进行的采集任务' })
      }
      break

    default:
      sendResponse({ success: false, error: 'Unknown message type' })
  }

  return true // 保持消息通道开启
})

console.log('[Content Script] 内容脚本已加载')
