const fs = require('fs')
const path = require('path')

// Configuration
const CONFIG = {
  updatesJsonPath: path.join(__dirname, '../dist/updates.json'),
  distDir: path.join(__dirname, '../dist'),
  baseUrl: 'https://zxhgs.com/extension-latest'
}

function generateUpdatesJson() {
  console.log('Generating updates.json...')

  // 1. Find the latest release info file
  const files = fs.readdirSync(process.cwd())
  const releaseInfoFiles = files.filter(f => f.startsWith('release-info-v') && f.endsWith('.json'))

  if (releaseInfoFiles.length === 0) {
    console.error('Error: No release-info-v*.json files found. Please run package script first.')
    process.exit(1)
  }

  // Sort to get the latest version (simple string sort might not be enough for complex versions, but works for standard format)
  // Better to parse versions, but assuming the file naming convention holds
  releaseInfoFiles.sort()
  const latestReleaseFile = releaseInfoFiles[releaseInfoFiles.length - 1]

  console.log(`Reading release info from: ${latestReleaseFile}`)
  let content = fs.readFileSync(latestReleaseFile, 'utf8')
  // Strip BOM if present
  if (content.charCodeAt(0) === 0xFEFF) {
    content = content.slice(1)
  }
  const releaseInfo = JSON.parse(content)

  // 2. Create updates.json content
  const updatesJson = {
    latest_version: releaseInfo.version,
    download_url: `${CONFIG.baseUrl}/${releaseInfo.filename}`,
    changelog_url: `${CONFIG.baseUrl}/CHANGELOG.html`,
    release_date: releaseInfo.build_date.split(' ')[0], // YYYY-MM-DD
    release_notes: `Version ${releaseInfo.version} released.`,
    min_required_version: '2.0.0', // Hardcoded or read from config if needed
    file_size_kb: releaseInfo.size_kb,
    sha256: releaseInfo.sha256
  }

  // 3. Ensure dist directory exists
  if (!fs.existsSync(CONFIG.distDir)) {
    fs.mkdirSync(CONFIG.distDir, { recursive: true })
  }

  // 4. Write updates.json
  fs.writeFileSync(CONFIG.updatesJsonPath, JSON.stringify(updatesJson, null, 2))
  console.log(`Successfully generated: ${CONFIG.updatesJsonPath}`)

  // 5. Move release info and zip to dist for easier deployment
  const zipPath = path.join(process.cwd(), releaseInfo.filename)
  const targetZipPath = path.join(CONFIG.distDir, releaseInfo.filename)

  if (fs.existsSync(zipPath)) {
    fs.copyFileSync(zipPath, targetZipPath)
    console.log(`Copied ${releaseInfo.filename} to dist/`)
  } else {
    console.warn(`Warning: Zip file ${releaseInfo.filename} not found in current directory.`)
  }
}

try {
  generateUpdatesJson()
} catch (error) {
  console.error('Failed to generate updates.json:', error)
  process.exit(1)
}
