/**
 * 内容脚本 - 在亚马逊页面中运行
 */

// 亚马逊站点配置（简化版，只在此处使用）
const AMAZON_SITES = [
  { id: 203, code: 'IE', name: '爱尔兰', currency: 'EUR', domain: 'amazon.ie' },
  { id: 1, code: 'US', name: '美国', currency: 'USD', domain: 'amazon.com' },
  { id: 2, code: 'UK', name: '英国', currency: 'GBP', domain: 'amazon.co.uk' },
  { id: 3, code: 'DE', name: '德国', currency: 'EUR', domain: 'amazon.de' },
  { id: 4, code: 'FR', name: '法国', currency: 'EUR', domain: 'amazon.fr' },
  { id: 5, code: 'IT', name: '意大利', currency: 'EUR', domain: 'amazon.it' },
  { id: 6, code: 'ES', name: '西班牙', currency: 'EUR', domain: 'amazon.es' },
  { id: 7, code: 'JP', name: '日本', currency: 'JPY', domain: 'amazon.co.jp' },
  { id: 8, code: 'CA', name: '加拿大', currency: 'CAD', domain: 'amazon.ca' },
  { id: 9, code: 'AU', name: '澳大利亚', currency: 'AUD', domain: 'amazon.com.au' },
  { id: 10, code: 'MX', name: '墨西哥', currency: 'MXN', domain: 'amazon.com.mx' },
  { id: 11, code: 'BR', name: '巴西', currency: 'BRL', domain: 'amazon.com.br' },
  { id: 12, code: 'IN', name: '印度', currency: 'INR', domain: 'amazon.in' },
  { id: 13, code: 'SG', name: '新加坡', currency: 'SGD', domain: 'amazon.sg' },
  { id: 14, code: 'AE', name: '阿联酋', currency: 'AED', domain: 'amazon.ae' },
  { id: 15, code: 'SA', name: '沙特阿拉伯', currency: 'SAR', domain: 'amazon.sa' },
  { id: 121, code: 'NL', name: '荷兰', currency: 'EUR', domain: 'amazon.nl' },
  { id: 122, code: 'PL', name: '波兰', currency: 'PLN', domain: 'amazon.pl' },
  { id: 123, code: 'SE', name: '瑞典', currency: 'SEK', domain: 'amazon.se' },
  { id: 124, code: 'BE', name: '比利时', currency: 'EUR', domain: 'amazon.com.be' }
];

function getSiteByHostname(hostname) {
  if (!hostname) return null;
  const host = hostname.toLowerCase();
  return AMAZON_SITES.find(site => host.includes(site.domain)) || null;
}

// ASIN 提取器
class AsinExtractor {
  static async extractFromPage() {
    const asins = new Set();
    const searchArea = document.querySelector('.sg-col-inner');
    const container = searchArea || document;

    container.querySelectorAll('[data-asin]').forEach(el => {
      const asin = el.getAttribute('data-asin');
      if (asin && this.isValidAsin(asin)) {
        asins.add(asin);
      }
    });

    const asinArray = Array.from(asins);
    console.log(`[ASIN Extractor] 提取到 ${asinArray.length} 个唯一ASIN`);
    return asinArray;
  }

  static isValidAsin(asin) {
    if (!asin || typeof asin !== 'string') return false;
    asin = asin.trim();
    return /^[A-Z0-9]{10}$/.test(asin) && !/^\d{10}$/.test(asin);
  }
}

// 分页处理器
class PaginationHandler {
  static hasNextPage() {
    const nextButton = this.getNextPageButton();
    if (!nextButton) return false;

    return !(
      nextButton.classList.contains('s-pagination-disabled') ||
      nextButton.classList.contains('disabled') ||
      nextButton.hasAttribute('disabled') ||
      nextButton.getAttribute('aria-disabled') === 'true'
    );
  }

  static getNextPageButton() {
    const selectors = [
      'a.s-pagination-next',
      '.s-pagination-item.s-pagination-next',
      'a[aria-label="Next"]',
      'a[aria-label="Go to next page"]'
    ];

    for (const selector of selectors) {
      const button = document.querySelector(selector);
      if (button) return button;
    }
    return null;
  }

  static async goToNextPage() {
    const nextButton = this.getNextPageButton();
    if (!nextButton || !this.hasNextPage()) return false;

    try {
      const currentUrl = window.location.href;
      const currentPage = this.getCurrentPageNumber();
      nextButton.click();
      await this.waitForPageLoad(currentUrl, currentPage);
      return true;
    } catch (error) {
      console.error('[Pagination] 进入下一页失败:', error);
      return false;
    }
  }

  static getCurrentPageNumber() {
    const activeElement = document.querySelector('.s-pagination-item.s-pagination-selected');
    if (activeElement) {
      const pageNum = parseInt(activeElement.textContent.trim(), 10);
      if (!isNaN(pageNum)) return pageNum;
    }

    const urlParams = new URLSearchParams(window.location.search);
    const pageFromUrl = urlParams.get('page') || urlParams.get('p');
    if (pageFromUrl) {
      const pageNum = parseInt(pageFromUrl, 10);
      if (!isNaN(pageNum)) return pageNum;
    }

    return 1;
  }

  static async waitForPageLoad(previousUrl, previousPage) {
    const maxWaitTime = 10000;
    const checkInterval = 200;
    let elapsedTime = 0;

    return new Promise((resolve, reject) => {
      const checkPageChanged = setInterval(() => {
        elapsedTime += checkInterval;
        const currentUrl = window.location.href;
        const currentPage = this.getCurrentPageNumber();

        if (currentUrl !== previousUrl || currentPage > previousPage) {
          clearInterval(checkPageChanged);
          setTimeout(resolve, 2000);
          return;
        }

        if (elapsedTime >= maxWaitTime) {
          clearInterval(checkPageChanged);
          reject(new Error('页面加载超时'));
        }
      }, checkInterval);
    });
  }

  static getTotalPages() {
    const pageButtons = document.querySelectorAll('.s-pagination-item');
    let maxPage = 0;

    pageButtons.forEach(btn => {
      if (btn.classList.contains('s-pagination-previous') ||
          btn.classList.contains('s-pagination-next') ||
          btn.classList.contains('s-pagination-ellipsis')) {
        return;
      }

      const pageNum = parseInt(btn.textContent.trim(), 10);
      if (!isNaN(pageNum) && pageNum > maxPage) {
        maxPage = pageNum;
      }
    });

    if (maxPage > 0) return maxPage;

    const resultStats = document.querySelector('.s-desktop-toolbar, .s-result-count');
    if (resultStats) {
      const match = resultStats.textContent.match(/of\s+([\d,]+)\s+results/i);
      if (match) {
        const totalResults = parseInt(match[1].replace(/,/g, ''), 10);
        return Math.ceil(totalResults / 16);
      }
    }

    return null;
  }
}

// 爬取状态
let isScrapingActive = false;
let isPaused = false;
let isCancelled = false;
let collectedAsins = [];

// 发送消息助手
function sendMessage(type, data) {
  chrome.runtime.sendMessage({ type, data }, () => {
    if (chrome.runtime.lastError) {
      // 忽略错误
    }
  });
}

// 执行ASIN采集任务
async function startScraping() {
  if (isScrapingActive) {
    console.log('[Content Script] 采集任务已在进行中');
    return;
  }

  isScrapingActive = true;
  isPaused = false;
  isCancelled = false;
  collectedAsins = [];

  try {
    console.log('[Content Script] 开始采集');

    const { scrapingMode } = await chrome.storage.local.get('scrapingMode');
    const mode = scrapingMode || 'single';

    console.log(`[Content Script] 采集模式: ${mode}`);

    // 后台采集模式
    if (mode === 'background') {
      const pageInfo = getPageInfo();
      const totalPages = PaginationHandler.getTotalPages() || 1;

      chrome.runtime.sendMessage({
        type: 'START_BACKGROUND_SCRAPING',
        data: { baseUrl: window.location.href, totalPages, pageInfo }
      }, (response) => {
        if (chrome.runtime.lastError) {
          console.error('[Content Script] 发送后台采集请求失败:', chrome.runtime.lastError);
          isScrapingActive = false;
        } else if (response && response.success) {
          console.log('[Content Script] 后台采集已启动');
        }
      });

      isScrapingActive = false;
      return;
    }

    // 前台采集模式
    let currentPage = 1;
    const isSinglePage = mode === 'single';
    const totalPages = isSinglePage ? 1 : (PaginationHandler.getTotalPages() || '未知');

    sendMessage('SCRAPING_PROGRESS', {
      currentPage,
      totalPages,
      asinCount: 0,
      status: 'scraping'
    });

    // 采集第一页
    const firstPageAsins = await AsinExtractor.extractFromPage();

    if (firstPageAsins.length === 0) {
      throw new Error('页面上没有找到ASIN数据，请确认：\n1. 当前在亚马逊商品列表页\n2. 页面已完全加载\n3. 可以看到商品列表');
    }

    collectedAsins.push(...firstPageAsins);
    console.log(`[Content Script] 第${currentPage}页采集完成，获取 ${firstPageAsins.length} 个ASIN`);

    sendMessage('SCRAPING_PROGRESS', {
      currentPage,
      totalPages,
      asinCount: collectedAsins.length,
      status: 'scraping'
    });

    // 多页采集
    if (!isSinglePage) {
      while (PaginationHandler.hasNextPage() && !isCancelled) {
        while (isPaused && !isCancelled) {
          await new Promise(resolve => setTimeout(resolve, 1000));
        }

        if (isCancelled) break;

        currentPage++;
        const success = await PaginationHandler.goToNextPage();
        if (!success) break;

        const pageAsins = await AsinExtractor.extractFromPage();
        collectedAsins.push(...pageAsins);

        sendMessage('SCRAPING_PROGRESS', {
          currentPage,
          totalPages,
          asinCount: collectedAsins.length,
          status: isPaused ? 'paused' : 'scraping'
        });
      }
    }

    // 去重并获取页面信息
    const uniqueAsins = Array.from(new Set(collectedAsins));
    const pageInfo = getPageInfo();

    const messageType = isCancelled ? 'SCRAPING_CANCELLED' : 'SCRAPING_COMPLETE';
    sendMessage(messageType, {
      asins: uniqueAsins,
      totalCount: uniqueAsins.length,
      totalPages: currentPage,
      pageInfo
    });

    console.log(`[Content Script] 采集${isCancelled ? '已取消' : '完成'}，共采集 ${uniqueAsins.length} 个唯一ASIN`);
  } catch (error) {
    console.error('[Content Script] 采集过程中发生错误:', error);
    sendMessage('SCRAPING_ERROR', { error: error.message });
  } finally {
    isScrapingActive = false;
    isPaused = false;
    isCancelled = false;
  }
}

// 提取分类面包屑
function extractCategoryBreadcrumb(selectors) {
  for (const selector of selectors) {
    const container = document.querySelector(selector);
    if (!container) continue;

    const links = container.querySelectorAll('a');
    const breadcrumb = [];

    links.forEach((link) => {
      const name = link.textContent.trim();
      const href = link.getAttribute('href');

      if (name && href && !name.toLowerCase().includes('back to') && !name.toLowerCase().includes('返回')) {
        const nodeMatch = href.match(/node[=/](\d+)/);
        breadcrumb.push({
          name,
          id: nodeMatch ? nodeMatch[1] : null,
          url: href
        });
      }
    });

    if (breadcrumb.length > 0) return breadcrumb;
  }
  return [];
}

// 获取页面完整信息
function getPageInfo() {
  const info = {
    site: null,
    marketplace: null,
    url: window.location.href,
    category: null,
    categoryId: null,
    categoryPath: null,
    categoryBreadcrumb: [],
    searchKeyword: null,
    resultCount: null,
    totalResults: null,
    sellerId: null,
    sellerName: null,
    timestamp: new Date().toISOString()
  };

  try {
    // 站点信息
    const siteInfo = getSiteByHostname(window.location.hostname);
    if (siteInfo) {
      info.site = siteInfo.code;
      info.siteName = siteInfo.name;
      info.siteId = siteInfo.id;
      info.currency = siteInfo.currency;
    }

    const urlParams = new URLSearchParams(window.location.search);
    info.marketplace = urlParams.get('marketplaceID') || null;

    // 分类信息
    const breadcrumbSelectors = [
      '#wayfinding-breadcrumbs_feature_div ul',
      '.a-breadcrumb',
      '[data-component-type="s-breadcrumb"]'
    ];

    const breadcrumb = extractCategoryBreadcrumb(breadcrumbSelectors);
    if (breadcrumb.length > 0) {
      info.categoryBreadcrumb = breadcrumb;
      info.categoryPath = breadcrumb.map(item => item.name).join(' > ');
      info.category = breadcrumb[breadcrumb.length - 1].name;
      info.categoryId = breadcrumb[breadcrumb.length - 1].id;
    }

    // 备用：从搜索下拉框获取
    if (!info.category) {
      const categoryElement = document.querySelector('#searchDropdownBox option[selected]');
      if (categoryElement) {
        info.category = categoryElement.textContent.trim();
        info.categoryBreadcrumb = [{ name: info.category, id: null }];
      }
    }

    // 从URL参数获取分类ID
    if (!info.categoryId) {
      const rhParam = urlParams.get('rh');
      if (rhParam) {
        const nodeMatch = rhParam.match(/n:(\d+)/);
        if (nodeMatch) {
          info.categoryId = nodeMatch[1];
          if (!info.category) {
            info.category = `Category ${nodeMatch[1]}`;
            info.categoryBreadcrumb = [{ name: info.category, id: info.categoryId }];
          }
        }
      }
    }

    // 搜索关键词
    const searchBox = document.querySelector('#twotabsearchtextbox, input[name="field-keywords"]');
    if (searchBox && searchBox.value) {
      info.searchKeyword = searchBox.value.trim();
    }
    if (!info.searchKeyword) {
      info.searchKeyword = urlParams.get('k') || urlParams.get('keywords') || null;
    }

    // 卖家信息
    info.sellerId = urlParams.get('me') || null;
    const sellerLink = document.querySelector('a[href*="/sp?seller="]');
    if (sellerLink) {
      info.sellerName = sellerLink.textContent.trim();
    }

    // 结果数量
    const resultText = document.querySelector('.s-desktop-toolbar .sg-col-inner h1, .s-result-count');
    if (resultText) {
      const text = resultText.textContent;
      const match = text.match(/(\d+)-(\d+)\s+of\s+([\d,]+)\s+results/i);
      if (match) {
        info.resultCount = match[2];
        info.totalResults = match[3].replace(/,/g, '');
      }
    }

    console.log('[Content Script] 页面信息:', info);
  } catch (error) {
    console.error('[Content Script] 获取页面信息失败:', error);
  }

  return info;
}

// 消息监听
chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
  console.log('[Content Script] 收到消息:', message);

  const handlers = {
    'GET_PAGE_INFO': () => sendResponse({ success: true, data: getPageInfo() }),
    'START_SCRAPING': () => {
      startScraping();
      sendResponse({ success: true });
    },
    'PAUSE_SCRAPING': () => {
      if (isScrapingActive) {
        isPaused = true;
        sendResponse({ success: true, paused: true });
      } else {
        sendResponse({ success: false, error: '没有正在进行的采集任务' });
      }
    },
    'RESUME_SCRAPING': () => {
      if (isScrapingActive && isPaused) {
        isPaused = false;
        sendResponse({ success: true, resumed: true });
      } else {
        sendResponse({ success: false, error: '没有暂停的采集任务' });
      }
    },
    'CANCEL_SCRAPING': () => {
      if (isScrapingActive) {
        isCancelled = true;
        isPaused = false;
        sendResponse({ success: true, cancelled: true });
      } else {
        sendResponse({ success: false, error: '没有正在进行的采集任务' });
      }
    }
  };

  const handler = handlers[message.type];
  if (handler) {
    handler();
  } else {
    sendResponse({ success: false, error: 'Unknown message type' });
  }

  return true;
});

console.log('[Content Script] 内容脚本已加载');

