/**
 * 产品信息浮动面板
 * 在页面左侧显示当前页面的产品信息
 */

// 检查扩展 context 是否有效
function isExtensionContextValid() {
  try {
    return !!(chrome.runtime && chrome.runtime.id)
  } catch (error) {
    return false
  }
}

class ProductPanel {
  constructor() {
    this.panel = null
    this.products = []
    this.isVisible = false
    this.isMinimized = false

    this.init()
  }

  async init() {
    console.log('[Product Panel] 初始化产品面板...')
    this.createPanel()

    // 读取保存的显示状态（默认关闭）
    const savedState = await this.loadPanelState()
    console.log('[Product Panel] 保存的状态:', savedState)

    // 恢复保存的状态
    this.isVisible = savedState?.isVisible || false
    this.isMinimized = savedState?.isMinimized || false

    if (this.isVisible) {
      // 如果之前是显示的，恢复显示状态（不重复保存）
      this.panel.style.display = 'flex'
      this.loadProducts()
      console.log('[Product Panel] 已恢复显示状态')
    } else {
      // 默认隐藏
      this.panel.style.display = 'none'
    }

    // 如果是最小化状态，恢复最小化
    if (this.isMinimized && this.isVisible) {
      const bodyEl = this.panel.querySelector('#asinPanelBody')
      const breadcrumbEl = this.panel.querySelector('#asinCategoryBreadcrumb')
      const minimizeBtn = this.panel.querySelector('#asinPanelMinimize')

      if (bodyEl) bodyEl.style.display = 'none'
      if (breadcrumbEl) breadcrumbEl.style.display = 'none'
      if (minimizeBtn) {
        minimizeBtn.textContent = '+'
        minimizeBtn.title = '展开'
      }
      this.panel.style.height = 'auto'
    }

    // 监听页面变化
    this.observePageChanges()
  }

  /**
   * 加载面板状态
   */
  async loadPanelState() {
    // 检查扩展 context 是否有效
    if (!isExtensionContextValid()) {
      console.log('[Product Panel] 扩展 context 已失效，使用默认状态')
      return { isVisible: false, isMinimized: false }
    }

    try {
      const result = await chrome.storage.local.get(['productPanelState'])
      return result.productPanelState || { isVisible: false, isMinimized: false }
    } catch (error) {
      // 扩展重新加载后 chrome API 会失效，这是正常的，静默处理
      if (error.message && error.message.includes('Extension context invalidated')) {
        console.log('[Product Panel] 扩展已重新加载，使用默认状态')
        return { isVisible: false, isMinimized: false }
      }
      console.error('[Product Panel] 加载状态失败:', error)
      return { isVisible: false, isMinimized: false }
    }
  }

  /**
   * 保存面板状态
   */
  async savePanelState() {
    // 检查扩展 context 是否有效
    if (!isExtensionContextValid()) {
      console.log('[Product Panel] 扩展 context 已失效，跳过状态保存')
      return
    }

    try {
      await chrome.storage.local.set({
        productPanelState: {
          isVisible: this.isVisible,
          isMinimized: this.isMinimized
        }
      })
      console.log('[Product Panel] 状态已保存:', { isVisible: this.isVisible, isMinimized: this.isMinimized })
    } catch (error) {
      // 扩展重新加载后 chrome API 会失效，这是正常的，静默处理
      if (error.message && error.message.includes('Extension context invalidated')) {
        console.log('[Product Panel] 扩展已重新加载，跳过状态保存')
        return
      }
      console.error('[Product Panel] 保存状态失败:', error)
    }
  }

  createPanel() {
    // 创建浮动面板
    const panel = document.createElement('div')
    panel.id = 'asin-product-panel'
    panel.className = 'asin-panel-container'
    panel.innerHTML = `
      <div class="asin-panel-header">
        <div class="asin-panel-title">
          <span class="asin-panel-icon">📦</span>
          <span class="asin-panel-text">当前页面产品</span>
          <span class="asin-panel-count" id="asinPanelCount">0</span>
        </div>
        <div class="asin-panel-controls">
          <button class="asin-panel-btn" id="asinPanelRefresh" title="刷新">🔄</button>
          <button class="asin-panel-btn" id="asinPanelMinimize" title="最小化">−</button>
          <button class="asin-panel-btn" id="asinPanelClose" title="关闭">×</button>
        </div>
      </div>
      <div class="asin-category-breadcrumb" id="asinCategoryBreadcrumb" style="display: none;">
        <div class="breadcrumb-icon">📁</div>
        <div class="breadcrumb-text" id="breadcrumbText"></div>
      </div>
      <div class="asin-panel-body" id="asinPanelBody">
        <div class="asin-panel-loading">
          <div class="asin-spinner"></div>
          <span>加载中...</span>
        </div>
      </div>
    `

    // 添加样式
    const style = document.createElement('style')
    style.textContent = `
      .asin-panel-container {
        position: fixed;
        right: 20px;
        top: 80px;
        width: 360px;
        max-height: calc(100vh - 120px);
        background: white;
        border-radius: 12px;
        box-shadow: 0 8px 32px rgba(0, 0, 0, 0.15);
        z-index: 999999;
        font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
        display: flex;
        flex-direction: column;
        overflow: hidden;
        animation: slideInRight 0.3s ease-out;
      }

      @keyframes slideInRight {
        from {
          transform: translateX(400px);
          opacity: 0;
        }
        to {
          transform: translateX(0);
          opacity: 1;
        }
      }


      .asin-panel-header {
        display: flex;
        align-items: center;
        justify-content: space-between;
        padding: 12px 16px;
        background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
        color: white;
        border-radius: 12px 12px 0 0;
      }

      .asin-panel-title {
        display: flex;
        align-items: center;
        gap: 8px;
        font-weight: 600;
        font-size: 14px;
      }

      .asin-panel-icon {
        font-size: 18px;
      }

      .asin-panel-count {
        background: rgba(255, 255, 255, 0.2);
        padding: 2px 8px;
        border-radius: 12px;
        font-size: 12px;
      }

      .asin-panel-controls {
        display: flex;
        gap: 4px;
      }

      .asin-panel-btn {
        background: rgba(255, 255, 255, 0.2);
        border: none;
        color: white;
        width: 28px;
        height: 28px;
        border-radius: 6px;
        cursor: pointer;
        font-size: 16px;
        display: flex;
        align-items: center;
        justify-content: center;
        transition: all 0.2s;
      }

      .asin-panel-btn:hover {
        background: rgba(255, 255, 255, 0.3);
        transform: scale(1.1);
      }

      .asin-panel-body {
        flex: 1;
        overflow-y: auto;
        padding: 12px;
        background: #f8fafc;
      }

      .asin-panel-body::-webkit-scrollbar {
        width: 6px;
      }

      .asin-panel-body::-webkit-scrollbar-track {
        background: #f1f5f9;
      }

      .asin-panel-body::-webkit-scrollbar-thumb {
        background: #cbd5e1;
        border-radius: 3px;
      }

      .asin-panel-body::-webkit-scrollbar-thumb:hover {
        background: #94a3b8;
      }

      .asin-panel-loading {
        display: flex;
        flex-direction: column;
        align-items: center;
        justify-content: center;
        padding: 40px 20px;
        gap: 12px;
        color: #64748b;
      }

      .asin-spinner {
        width: 32px;
        height: 32px;
        border: 3px solid #e2e8f0;
        border-top-color: #667eea;
        border-radius: 50%;
        animation: spin 1s linear infinite;
      }

      @keyframes spin {
        to { transform: rotate(360deg); }
      }

      .asin-panel-empty {
        text-align: center;
        padding: 40px 20px;
        color: #64748b;
      }

      .asin-panel-empty-icon {
        font-size: 48px;
        margin-bottom: 12px;
        opacity: 0.5;
      }

      .asin-product-item {
        background: white;
        border: 1px solid #e2e8f0;
        border-radius: 8px;
        padding: 12px;
        margin-bottom: 12px;
        transition: all 0.2s;
      }

      .asin-product-item:hover {
        border-color: #667eea;
        box-shadow: 0 4px 12px rgba(102, 126, 234, 0.1);
        transform: translateY(-2px);
      }

      .asin-product-header {
        display: flex;
        align-items: center;
        gap: 8px;
        margin-bottom: 10px;
        padding-bottom: 8px;
        border-bottom: 1px solid #e2e8f0;
      }

      .asin-product-index {
        display: flex;
        align-items: center;
        justify-content: center;
        width: 24px;
        height: 24px;
        background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
        color: white;
        border-radius: 50%;
        font-size: 10px;
        font-weight: 600;
        flex-shrink: 0;
      }

      .asin-product-asin {
        flex: 1;
        font-size: 11px;
        color: #64748b;
      }

      .asin-product-asin-code {
        color: #667eea;
        font-weight: 600;
        font-family: 'Consolas', 'Monaco', monospace;
      }

      .asin-product-image {
        width: 100%;
        height: 80px;
        display: flex;
        align-items: center;
        justify-content: center;
        background: #f8fafc;
        border-radius: 6px;
        margin-bottom: 8px;
        overflow: hidden;
      }

      .asin-product-image img {
        max-width: 100%;
        max-height: 100%;
        object-fit: contain;
      }

      .asin-product-title {
        font-size: 12px;
        line-height: 1.4;
        color: #1e293b;
        margin-bottom: 8px;
        display: -webkit-box;
        -webkit-line-clamp: 2;
        -webkit-box-orient: vertical;
        overflow: hidden;
      }

      .asin-product-info {
        display: flex;
        align-items: center;
        justify-content: space-between;
        margin-bottom: 8px;
      }

      .asin-product-price {
        font-size: 14px;
        font-weight: 700;
        color: #ef4444;
      }

      .asin-product-rating {
        display: flex;
        align-items: center;
        gap: 4px;
        font-size: 11px;
      }

      .asin-rating-stars {
        color: #f59e0b;
        font-weight: 600;
      }

      .asin-rating-count {
        color: #64748b;
      }

      .asin-product-link {
        display: flex;
        align-items: center;
        justify-content: center;
        gap: 4px;
        width: 100%;
        padding: 6px 12px;
        background: #667eea;
        color: white;
        text-decoration: none;
        border-radius: 6px;
        font-size: 11px;
        font-weight: 600;
        transition: all 0.2s;
      }

      .asin-product-link:hover {
        background: #818cf8;
        transform: translateY(-1px);
      }

      .asin-category-breadcrumb {
        padding: 10px 16px;
        background: linear-gradient(135deg, #f8fafc 0%, #e0e7ff 100%);
        border-bottom: 1px solid #e2e8f0;
        display: flex;
        align-items: flex-start;
        gap: 8px;
      }

      .breadcrumb-icon {
        font-size: 14px;
        flex-shrink: 0;
        margin-top: 2px;
      }

      .breadcrumb-text {
        flex: 1;
        font-size: 11px;
        line-height: 1.5;
        color: #475569;
        word-break: break-word;
      }

      .breadcrumb-separator {
        color: #94a3b8;
        margin: 0 4px;
      }
    `

    document.head.appendChild(style)
    document.body.appendChild(panel)
    this.panel = panel

    // 绑定事件
    this.bindEvents()
  }

  bindEvents() {
    // 刷新按钮
    const refreshBtn = this.panel.querySelector('#asinPanelRefresh')
    refreshBtn.addEventListener('click', () => {
      this.loadProducts()
    })

    // 最小化按钮
    const minimizeBtn = this.panel.querySelector('#asinPanelMinimize')
    minimizeBtn.addEventListener('click', () => {
      this.toggleMinimize()
    })

    // 关闭按钮
    const closeBtn = this.panel.querySelector('#asinPanelClose')
    closeBtn.addEventListener('click', () => {
      this.hide()
    })
  }

  async loadProducts() {
    console.log('[Product Panel] 开始加载产品...')

    const bodyEl = this.panel.querySelector('#asinPanelBody')
    bodyEl.innerHTML = `
      <div class="asin-panel-loading">
        <div class="asin-spinner"></div>
        <span>加载中...</span>
      </div>
    `

    try {
      // 等待一小段时间确保页面加载完成
      await new Promise(resolve => setTimeout(resolve, 500))

      // 提取分类面包屑
      this.extractCategoryBreadcrumb()

      // 提取产品信息
      this.products = await this.extractProducts()

      console.log(`[Product Panel] 提取到 ${this.products.length} 个产品`)

      // 更新计数
      const countEl = this.panel.querySelector('#asinPanelCount')
      countEl.textContent = this.products.length

      // 显示产品
      this.displayProducts()

    } catch (error) {
      console.error('[Product Panel] 加载产品失败:', error)
      bodyEl.innerHTML = `
        <div class="asin-panel-empty">
          <div class="asin-panel-empty-icon">⚠️</div>
          <div>加载失败，请重试</div>
        </div>
      `
    }
  }

  extractCategoryBreadcrumb() {
    const breadcrumbEl = this.panel.querySelector('#asinCategoryBreadcrumb')
    const breadcrumbTextEl = this.panel.querySelector('#breadcrumbText')

    try {
      console.log('[Product Panel] 提取分类面包屑...')

      let categoryPath = []

      // 方法1: 从顶部面包屑导航获取（最可靠，优先级最高）
      const breadcrumbSelectors = [
        '#wayfinding-breadcrumbs_feature_div ul',
        '.a-breadcrumb',
        '[data-component-type="s-breadcrumb"]',
        '#nav-subnav[data-category]'
      ]

      for (const selector of breadcrumbSelectors) {
        const breadcrumbContainer = document.querySelector(selector)
        if (breadcrumbContainer) {
          console.log('[Product Panel] 找到面包屑导航:', selector)
          const links = breadcrumbContainer.querySelectorAll('a')

          links.forEach(link => {
            const text = link.textContent.trim()
            const href = link.getAttribute('href')
            // 排除"返回搜索结果"等非分类链接
            if (text && href &&
                !text.toLowerCase().includes('back to') &&
                !text.toLowerCase().includes('返回') &&
                text !== 'Home' &&
                text !== 'All') {
              categoryPath.push(text)
            }
          })

          if (categoryPath.length > 0) {
            console.log('[Product Panel] 面包屑路径:', categoryPath)
            breadcrumbTextEl.innerHTML = categoryPath.join(' <span class="breadcrumb-separator">›</span> ')
            breadcrumbEl.style.display = 'flex'
            return
          }
        }
      }

      // 方法2: 从左侧导航栏获取完整的分类路径（包括父分类和选中的子分类）
      const departmentSection = document.querySelector('#departments')
      if (departmentSection) {
        // 获取所有加粗的链接（表示当前选中或父级分类）
        const boldLinks = departmentSection.querySelectorAll('.a-text-bold')

        console.log(`[Product Panel] 找到 ${boldLinks.length} 个加粗分类`)

        categoryPath = []
        boldLinks.forEach((link, index) => {
          const text = link.textContent.trim()
          if (text && text !== 'Department' && text !== 'Any Department') {
            console.log(`[Product Panel] 加粗分类 ${index + 1}:`, text)
            categoryPath.push(text)
          }
        })

        if (categoryPath.length > 0) {
          console.log('[Product Panel] 左侧导航分类路径:', categoryPath)
          breadcrumbTextEl.innerHTML = categoryPath.join(' <span class="breadcrumb-separator">›</span> ')
          breadcrumbEl.style.display = 'flex'
          return
        }
      }

      // 方法3: 从搜索下拉框获取
      const categoryElement = document.querySelector('#searchDropdownBox option[selected]')
      if (categoryElement) {
        const category = categoryElement.textContent.trim()
        if (category && category !== 'All Departments') {
          console.log('[Product Panel] 搜索下拉框分类:', category)
          breadcrumbTextEl.textContent = category
          breadcrumbEl.style.display = 'flex'
          return
        }
      }

      // 方法4: 从URL参数获取搜索关键词
      const urlParams = new URLSearchParams(window.location.search)
      const searchKeyword = urlParams.get('k')

      if (searchKeyword) {
        console.log('[Product Panel] 搜索关键词:', searchKeyword)
        breadcrumbTextEl.innerHTML = `<strong>🔍 搜索:</strong> ${searchKeyword}`
        breadcrumbEl.style.display = 'flex'
        return
      }

      // 方法5: 从URL路径判断（如 /gp/bestsellers, /gp/new-releases）
      const pathname = window.location.pathname
      if (pathname.includes('/gp/bestsellers')) {
        breadcrumbTextEl.textContent = '🏆 Best Sellers'
        breadcrumbEl.style.display = 'flex'
        return
      } else if (pathname.includes('/gp/new-releases')) {
        breadcrumbTextEl.textContent = '✨ New Releases'
        breadcrumbEl.style.display = 'flex'
        return
      }

      // 如果都没有找到，隐藏面包屑
      breadcrumbEl.style.display = 'none'
      console.log('[Product Panel] 未找到分类信息')

    } catch (error) {
      console.error('[Product Panel] 提取分类面包屑失败:', error)
      breadcrumbEl.style.display = 'none'
    }
  }

  async extractProducts() {
    const products = []

    // 查找所有商品元素
    const productElements = document.querySelectorAll('[data-component-type="s-search-result"]')

    console.log(`[Product Panel] 找到 ${productElements.length} 个商品元素`)

    productElements.forEach((element, index) => {
      try {
        const asin = element.getAttribute('data-asin')

        if (!asin || asin === '') {
          return
        }

        const product = {
          asin: asin,
          title: null,
          price: null,
          currency: null,
          image: null,
          rating: null,
          reviews: null,
          url: null
        }

        // 提取标题
        const titleEl = element.querySelector('h2 a span, h2 span')
        if (titleEl) {
          product.title = titleEl.textContent.trim()
        }

        // 提取价格
        const priceEl = element.querySelector('.a-price .a-offscreen')
        if (priceEl) {
          product.price = priceEl.textContent.trim()
        }

        // 提取图片
        const imageEl = element.querySelector('img.s-image')
        if (imageEl) {
          product.image = imageEl.src || imageEl.getAttribute('data-image-latency')
        }

        // 提取评分
        const ratingEl = element.querySelector('[aria-label*="out of"]')
        if (ratingEl) {
          const ratingText = ratingEl.getAttribute('aria-label')
          const match = ratingText.match(/(\d+\.?\d*)/)
          if (match) {
            product.rating = parseFloat(match[1])
          }
        }

        // 提取评论数
        const reviewsEl = element.querySelector('[aria-label*="ratings"]')
        if (reviewsEl) {
          const reviewsText = reviewsEl.getAttribute('aria-label')
          const match = reviewsText.match(/([\d,]+)/)
          if (match) {
            product.reviews = match[1]
          }
        }

        // 生成URL
        product.url = `https://${window.location.hostname}/dp/${asin}`

        // 只添加有标题的产品
        if (product.title) {
          products.push(product)
        }

      } catch (error) {
        console.warn(`[Product Panel] 提取产品 ${index} 失败:`, error)
      }
    })

    return products
  }

  displayProducts() {
    const bodyEl = this.panel.querySelector('#asinPanelBody')

    if (this.products.length === 0) {
      bodyEl.innerHTML = `
        <div class="asin-panel-empty">
          <div class="asin-panel-empty-icon">📦</div>
          <div>当前页面没有产品</div>
        </div>
      `
      return
    }

    bodyEl.innerHTML = ''

    this.products.forEach((product, index) => {
      const productEl = document.createElement('div')
      productEl.className = 'asin-product-item'

      productEl.innerHTML = `
        <div class="asin-product-header">
          <div class="asin-product-index">${index + 1}</div>
          <div class="asin-product-asin">
            ASIN: <span class="asin-product-asin-code">${product.asin}</span>
          </div>
        </div>
        ${product.image ? `
          <div class="asin-product-image">
            <img src="${product.image}" alt="${product.title || 'Product'}" />
          </div>
        ` : ''}
        ${product.title ? `
          <div class="asin-product-title">${product.title}</div>
        ` : ''}
        <div class="asin-product-info">
          ${product.price ? `
            <div class="asin-product-price">${product.price}</div>
          ` : '<div></div>'}
          ${product.rating ? `
            <div class="asin-product-rating">
              <span class="asin-rating-stars">⭐ ${product.rating}</span>
              ${product.reviews ? `<span class="asin-rating-count">(${product.reviews})</span>` : ''}
            </div>
          ` : ''}
        </div>
        <a href="${product.url}" target="_blank" class="asin-product-link">
          查看详情 ↗
        </a>
      `

      bodyEl.appendChild(productEl)
    })
  }

  toggleMinimize() {
    this.isMinimized = !this.isMinimized
    const minimizeBtn = this.panel.querySelector('#asinPanelMinimize')
    const bodyEl = this.panel.querySelector('#asinPanelBody')
    const breadcrumbEl = this.panel.querySelector('#asinCategoryBreadcrumb')

    if (this.isMinimized) {
      // 最小化：隐藏内容区域
      bodyEl.style.display = 'none'
      if (breadcrumbEl) {
        breadcrumbEl.style.display = 'none'
      }
      minimizeBtn.textContent = '+'
      minimizeBtn.title = '展开'
      this.panel.style.height = 'auto'
    } else {
      // 展开：显示内容区域
      bodyEl.style.display = 'block'
      if (breadcrumbEl && breadcrumbEl.querySelector('#breadcrumbText').textContent) {
        breadcrumbEl.style.display = 'flex'
      }
      minimizeBtn.textContent = '−'
      minimizeBtn.title = '最小化'
      this.panel.style.height = ''
    }

    this.savePanelState()  // 保存状态
  }

  show() {
    this.panel.style.display = 'flex'
    this.isVisible = true
    this.loadProducts()
    this.savePanelState()  // 保存状态
  }

  hide() {
    // 立即更新状态，而不是等待动画完成
    this.isVisible = false
    this.savePanelState()  // 保存状态

    this.panel.style.animation = 'slideInRight 0.3s ease-out reverse'
    setTimeout(() => {
      this.panel.style.display = 'none'
    }, 300)
  }

  observePageChanges() {
    // 监听URL变化
    let lastUrl = window.location.href

    const checkUrlChange = () => {
      const currentUrl = window.location.href
      if (currentUrl !== lastUrl) {
        lastUrl = currentUrl
        console.log('[Product Panel] URL changed, reloading products...')
        if (this.isVisible) {
          this.loadProducts()
        }
      }
    }

    setInterval(checkUrlChange, 2000)
  }
}

// 初始化产品面板
let productPanel = null

// 页面加载完成后初始化（不自动显示，由用户保存的状态决定）
if (document.readyState === 'loading') {
  document.addEventListener('DOMContentLoaded', () => {
    setTimeout(() => {
      productPanel = new ProductPanel()
      // 移除自动显示逻辑，由 init() 中的状态恢复决定
    }, 1000)
  })
} else {
  setTimeout(() => {
    productPanel = new ProductPanel()
    // 移除自动显示逻辑，由 init() 中的状态恢复决定
  }, 1000)
}

// 监听来自popup的消息
chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
  // 检查扩展 context 是否仍然有效
  if (!isExtensionContextValid()) {
    console.log('[Product Panel] 扩展 context 已失效，忽略消息')
    return false
  }

  console.log('[Product Panel] 收到消息:', message.type)

  if (message.type === 'TOGGLE_PRODUCT_PANEL') {
    if (!productPanel) {
      console.log('[Product Panel] 创建新的面板实例')
      productPanel = new ProductPanel()
    }

    console.log('[Product Panel] 当前状态 isVisible:', productPanel.isVisible)

    if (productPanel.isVisible) {
      console.log('[Product Panel] 隐藏面板')
      productPanel.hide()
      // 确保在响应中返回正确的状态
      console.log('[Product Panel] 隐藏后状态:', productPanel.isVisible)
      sendResponse({ success: true, isVisible: productPanel.isVisible })
    } else {
      console.log('[Product Panel] 显示面板')
      productPanel.show()
      // 确保在响应中返回正确的状态
      console.log('[Product Panel] 显示后状态:', productPanel.isVisible)
      sendResponse({ success: true, isVisible: productPanel.isVisible })
    }
  } else if (message.type === 'GET_PANEL_STATE') {
    // 返回当前面板状态
    const state = productPanel ? productPanel.isVisible : false
    console.log('[Product Panel] 返回当前状态:', state)
    sendResponse({
      success: true,
      isVisible: state
    })
  }

  return true
})

console.log('[Product Panel] 产品面板脚本已加载')

