/**
 * API 工具模块 - 统一的 API 调用封装
 */

const DEFAULT_TIMEOUT = 30000;

/**
 * 发送 HTTP 请求（带超时和错误处理）
 */
export async function fetchWithTimeout(url, options = {}, timeout = DEFAULT_TIMEOUT) {
  const controller = new AbortController();
  const timeoutId = setTimeout(() => controller.abort(), timeout);

  try {
    const response = await fetch(url, {
      ...options,
      signal: controller.signal
    });
    clearTimeout(timeoutId);
    return response;
  } catch (error) {
    clearTimeout(timeoutId);
    if (error.name === 'AbortError') {
      throw new Error(`请求超时（${timeout / 1000}秒）`);
    }
    throw error;
  }
}

/**
 * 发送 JSON 请求
 */
export async function fetchJSON(url, data = null, method = 'GET', timeout = DEFAULT_TIMEOUT) {
  const options = {
    method,
    headers: { 'Content-Type': 'application/json' }
  };

  if (data && method !== 'GET') {
    options.body = JSON.stringify(data);
  }

  const response = await fetchWithTimeout(url, options, timeout);

  if (!response.ok) {
    const errorText = await response.text().catch(() => '无法读取错误信息');
    throw new Error(`API返回错误 ${response.status}: ${response.statusText}\n${errorText}`);
  }

  return response.json().catch(() => {
    throw new Error('服务器响应格式错误');
  });
}

/**
 * 上传 ASIN 到后端 API
 */
export async function uploadAsins(asins, pageInfo, config) {
  const { endpoint, collectedBy = 'browser-extension' } = config;

  const uploadData = {
    asins,
    site: pageInfo.site || 'US',
    collectedBy,
    batchName: `Browser Collection ${new Date().toISOString().split('T')[0]}`,
    pageInfo: {
      ...pageInfo,
      categoryPath: pageInfo.categoryPath || pageInfo.category,
      categoryBreadcrumb: pageInfo.categoryBreadcrumb || [],
      categoryId: pageInfo.categoryId || null,
      searchKeyword: pageInfo.searchKeyword || null,
      url: pageInfo.url || null
    },
    extraData: {
      timestamp: new Date().toISOString(),
      source: 'amazon-asin-scraper-v2',
      version: chrome.runtime.getManifest().version
    }
  };

  return fetchJSON(endpoint, uploadData, 'POST');
}

/**
 * 查询商家统计
 */
export async function querySellerStats(endpoint, minAsinCount, minOfferCount) {
  const baseUrl = endpoint.replace('/api/upload-asins', '');
  const queryUrl = `${baseUrl}/api/seller-stats`;
  const params = new URLSearchParams({
    minAsinCount: minAsinCount.toString(),
    minOfferCount: minOfferCount.toString()
  });

  return fetchJSON(`${queryUrl}?${params.toString()}`);
}

/**
 * 获取商家详情页面 HTML
 */
export async function fetchSellerDetail(url) {
  const response = await fetchWithTimeout(url, { method: 'GET' });

  if (!response.ok) {
    throw new Error(`HTTP错误 ${response.status}: ${response.statusText}`);
  }

  return response.text();
}

/**
 * 解析商家详情 HTML
 */
export function parseSellerDetail(html) {
  const data = {
    businessName: null,
    businessType: null,
    tradeRegisterNumber: null,
    phone: null,
    email: null,
    address: []
  };

  const extractField = (patterns, content) => {
    for (const pattern of patterns) {
      const match = content.match(pattern);
      if (match && match[1]) return match[1].trim();
    }
    return null;
  };

  const sellerInfoMatch = html.match(/<div[^>]*id="page-section-detail-seller-info"[^>]*>([\s\S]*?)<\/div>\s*<\/div>\s*<\/div>\s*<\/div>/i);
  const content = sellerInfoMatch ? sellerInfoMatch[1] : html;

  // 提取各字段
  data.businessName = extractField([
    /Business Name:\s*<\/span>\s*<span[^>]*>([^<]+)<\/span>/i,
    /<span[^>]*class="[^"]*a-text-bold[^"]*"[^>]*>Business Name:\s*<\/span>\s*<span[^>]*>([^<]+)<\/span>/i
  ], content);

  data.businessType = extractField([
    /Business Type:\s*<\/span>\s*<span[^>]*>([^<]+)<\/span>/i,
    /<span[^>]*class="[^"]*a-text-bold[^"]*"[^>]*>Business Type:\s*<\/span>\s*<span[^>]*>([^<]+)<\/span>/i
  ], content);

  data.tradeRegisterNumber = extractField([
    /Trade Register Number:\s*<\/span>\s*<span[^>]*>([^<]+)<\/span>/i
  ], content);

  data.phone = extractField([
    /Phone number:\s*<\/span>\s*<span[^>]*>([^<]+)<\/span>/i
  ], content);

  data.email = extractField([
    /Email:\s*<\/span>\s*<span[^>]*>([^<]+)<\/span>/i
  ], content);

  // 提取地址
  const addressPattern = /Business Address:\s*<\/span>\s*<\/div>([\s\S]*?)(?:This seller|<\/div>\s*<\/div>\s*<\/div>)/i;
  const addressSection = content.match(addressPattern);
  if (addressSection) {
    const addressMatches = addressSection[1].matchAll(/<div[^>]*class="[^"]*indent-left[^"]*"[^>]*>\s*<span[^>]*>([^<]+)<\/span>\s*<\/div>/gi);
    for (const match of addressMatches) {
      const line = match[1].trim();
      if (line && !line.match(/^This seller/i)) {
        data.address.push(line);
      }
    }
  }

  return data;
}

