/**
 * Storage 工具模块 - 统一的存储操作封装
 */

/**
 * 获取存储的值
 */
export async function getStorage(keys) {
  try {
    return await chrome.storage.local.get(keys);
  } catch (error) {
    console.error('[Storage] 获取失败:', error);
    return Array.isArray(keys) ? {} : { [keys]: undefined };
  }
}

/**
 * 设置存储的值
 */
export async function setStorage(items) {
  try {
    await chrome.storage.local.set(items);
    return true;
  } catch (error) {
    console.error('[Storage] 设置失败:', error);
    return false;
  }
}

/**
 * 删除存储的值
 */
export async function removeStorage(keys) {
  try {
    await chrome.storage.local.remove(keys);
    return true;
  } catch (error) {
    console.error('[Storage] 删除失败:', error);
    return false;
  }
}

/**
 * 获取用户信息
 */
export async function getCurrentUser() {
  const { currentUser } = await getStorage('currentUser');
  return currentUser || null;
}

/**
 * 检查登录状态
 */
export async function checkAuth() {
  const { isAuthenticated, loginTime, currentUser } = await getStorage([
    'isAuthenticated',
    'loginTime',
    'currentUser'
  ]);

  if (!isAuthenticated || !currentUser) {
    return { authenticated: false, expired: false };
  }

  const expired = loginTime && (Date.now() - loginTime) > 24 * 60 * 60 * 1000;
  return { authenticated: true, expired, user: currentUser };
}

/**
 * 保存登录信息
 */
export async function saveAuth(user) {
  return setStorage({
    isAuthenticated: true,
    currentUser: user,
    loginTime: Date.now()
  });
}

/**
 * 清除登录信息
 */
export async function clearAuth() {
  return removeStorage(['isAuthenticated', 'currentUser', 'loginTime']);
}

/**
 * 获取 API 配置
 */
export async function getApiConfig() {
  const { apiEndpoint, scrapingMode, concurrency } = await getStorage([
    'apiEndpoint',
    'scrapingMode',
    'concurrency'
  ]);

  return {
    endpoint: apiEndpoint || 'https://njdk.xyz:18012/api/upload-asins',
    scrapingMode: scrapingMode || 'single',
    concurrency: concurrency || 5
  };
}

/**
 * 保存 API 配置
 */
export async function saveApiConfig(config) {
  return setStorage(config);
}

/**
 * 获取面板状态
 */
export async function getPanelState() {
  const { productPanelState } = await getStorage('productPanelState');
  return productPanelState || { isVisible: false, isMinimized: false };
}

/**
 * 保存面板状态
 */
export async function savePanelState(state) {
  return setStorage({ productPanelState: state });
}

